package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal

import android.graphics.Rect
import android.view.View
import androidx.lifecycle.Lifecycle
import androidx.lifecycle.LifecycleEventObserver
import androidx.lifecycle.LifecycleOwner
import androidx.lifecycle.findViewTreeLifecycleOwner
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import kotlinx.coroutines.channels.ProducerScope
import kotlinx.coroutines.channels.awaitClose
import kotlinx.coroutines.delay
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.callbackFlow
import kotlinx.coroutines.flow.channelFlow
import kotlinx.coroutines.flow.collectLatest
import kotlinx.coroutines.flow.conflate
import kotlinx.coroutines.flow.distinctUntilChanged
import kotlinx.coroutines.flow.flow
import kotlinx.coroutines.flow.flowOn

/**
 * Use this approach to get/listen to real state of visibility of the view.
 *
 * It means, it checks whether activity is resumed, view isShows, view's area is on-the-screen.
 *
 * Feel free to improve/extend/optimize it.
 */
fun interface ViewVisibilityTracker {
    fun isVisibleFlow(view: View): Flow<Boolean>
}

internal class ViewVisibilityTrackerImpl : ViewVisibilityTracker {
    override fun isVisibleFlow(view: View): Flow<Boolean> = channelFlow {
        // Let's start listening to view's window attached state.
        collectLatestIsAttachedWindow(view)
    }.conflateDistinctMain()
}

private suspend fun ProducerScope<Boolean>.collectLatestIsAttachedWindow(view: View) {
    view.isAttachedToWindowFlow().collectLatest { isAttachedToWindow ->
        if (isAttachedToWindow) {
            // Once view is attached to window,
            // it makes sense now to check for lifecycle pause/resume state (if lifecycle available),
            // since if not attached, view most likely doesn't have a lifecycle owner.
            collectLatestIsLifecycleResumed(view)
        } else {
            // If not attached to window, consider view not visible.
            send(false)
        }
    }
}

private suspend fun ProducerScope<Boolean>.collectLatestIsLifecycleResumed(view: View) {
    view.findViewTreeLifecycleOwner().isLifecycleResumedFlow().collectLatest { isLifecycleResumed ->
        // Once lifecycle is resumed or absent (99% unity env case),
        // let's start checking for view visible area.
        if (isLifecycleResumed != false) {
            collectLatestIsEnoughAreaVisible(view)
        } else {
            // lifecycle is paused, consider view not visible.
            send(false)
        }
    }
}

private suspend fun ProducerScope<Boolean>.collectLatestIsEnoughAreaVisible(view: View) {
    view.isEnoughAreaVisibleFlow().collectLatest {
        send(it)
    }
}

private fun View.isAttachedToWindowFlow(): Flow<Boolean> = callbackFlow {
    // Sending an initial value first.
    send(isAttachedToWindow)

    val listener = object : View.OnAttachStateChangeListener {
        override fun onViewAttachedToWindow(p0: View) {
            // Flow is setup as conflated, therefore,
            // trySend() is practically successful every call
            // unless channel closed/coroutine cancelled.
            trySend(true)
        }

        override fun onViewDetachedFromWindow(p0: View) {
            trySend(false)
        }
    }
    addOnAttachStateChangeListener(listener)

    awaitClose {
        removeOnAttachStateChangeListener(listener)
    }
}.conflateDistinctMain()

private fun LifecycleOwner?.isLifecycleResumedFlow(): Flow<Boolean?> {
    val lifecycle = this?.lifecycle
        // lifecycle is unavailable,
        // e.g. when Unity env uses Activity instead of Compat Activity,
        // or view not attached to window.
        ?: return flow {
            emit(null)
        }

    return callbackFlow {
        val lifecycleEventObserver = LifecycleEventObserver { _, event ->
            when (event) {
                Lifecycle.Event.ON_PAUSE -> trySend(false)
                Lifecycle.Event.ON_RESUME -> trySend(true)
                else -> {
                    /* no-op */
                }
            }
        }
        // An initial value is practically emitted automatically right after the subscription.
        lifecycle.addObserver(lifecycleEventObserver)

        awaitClose {
            lifecycle.removeObserver(lifecycleEventObserver)
        }
    }.conflateDistinctMain()
}

private fun View.isEnoughAreaVisibleFlow(): Flow<Boolean> = flow {
    val rect = Rect(0, 0, 0, 0)
    // I use polling because View's scroll change listeners do not work for Jetpack compose based scrollable containers,
    // e.g Column(Modifier.scrollable()).
    // This way, though not pretty or well optimized,
    // the solution works for both View and Jetpack Compose frameworks.
    while (true) {
        // Let's make sure at least something is visible on the screen,
        // TODO. Overlap view check if necessary; might be tricky in Jetpack Compose.
        emit(isShown && getGlobalVisibleRect(rect))
        delay(500)
    }
}.conflateDistinctMain()

private fun <T> Flow<T>.conflateDistinctMain() =
    conflate().distinctUntilChanged().flowOn(DispatcherProvider().main)
