package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.media.stream

import kotlinx.coroutines.channels.awaitClose
import kotlinx.coroutines.flow.Flow
import kotlinx.coroutines.flow.callbackFlow
import java.io.File

/**
 * Flow based listener for the media stream lifecycle.
 */
internal class MediaStreamListenerFlow(initialStatus: MediaStreamStatus) : MediaStreamListener {

    private var _lastStreamStatus: MediaStreamStatus = initialStatus
    override val lastStreamStatus: MediaStreamStatus
        get() = _lastStreamStatus

    val streamStatusFlow: Flow<MediaStreamStatus> = callbackFlow {
        onStreamStarted = { file: File ->
            trySend(MediaStreamStatus.InProgress(file, MediaStreamStatus.Progress(0, 0)))
        }
        onStreamProgress = { file: File, progress: MediaStreamStatus.Progress ->
            trySend(MediaStreamStatus.InProgress(file, progress))
        }
        onStreamComplete = { complete ->
            trySend(complete)
        }
        onStreamError = { error ->
            trySend(error)
        }
        awaitClose {
            onStreamStarted = null
            onStreamProgress = null
            onStreamComplete = null
            onStreamError = null
        }
    }

    private var onStreamStarted: ((File) -> Unit)? = null
    private var onStreamProgress: ((File, MediaStreamStatus.Progress) -> Unit)? = null
    private var onStreamComplete: ((MediaStreamStatus.Complete) -> Unit)? = null
    private var onStreamError: ((MediaStreamStatus.Failure) -> Unit)? = null

    override fun onStreamStarted(file: File) {
        _lastStreamStatus = MediaStreamStatus.InProgress(file, MediaStreamStatus.Progress(0, 0))
        onStreamStarted?.invoke(file)
    }

    override fun onStreamProgress(file: File, progress: MediaStreamStatus.Progress) {
        _lastStreamStatus = MediaStreamStatus.InProgress(file, progress)
        onStreamProgress?.invoke(file, progress)
    }

    override fun onStreamComplete(result: MediaStreamStatus.Complete) {
        _lastStreamStatus = result
        onStreamComplete?.invoke(result)
    }

    override fun onStreamError(error: MediaStreamStatus.Failure) {
        _lastStreamStatus = error
        onStreamError?.invoke(error)
    }
}