package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.mraid

import android.annotation.SuppressLint
import android.content.Context
import androidx.webkit.WebViewClientCompat
import com.moloco.sdk.BuildConfig
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.Result
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.BaseWebView
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.applyCSSRenderingFix
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.MraidAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.loadDataWithDefaultBaseUrl
import kotlinx.coroutines.coroutineScope
import kotlinx.coroutines.flow.combine
import kotlinx.coroutines.flow.first
import kotlinx.coroutines.withContext

// TODO. Ugly. Duplication of VastWebView.
//  Review settings during webview build/init.
@SuppressLint("SetJavaScriptEnabled", "ViewConstructor")
internal class MraidWebView(
    context: Context,
    mraidJsCommandUrlSource: MraidJsCommandUrlSource,
) : BaseWebView(context) {

    init {
        setWebContentsDebuggingEnabled(BuildConfig.DEBUG)

        scrollBarStyle = SCROLLBARS_INSIDE_OVERLAY
        isHorizontalScrollBarEnabled = false
        isVerticalScrollBarEnabled = false
        isScrollContainer = false

        with(settings) {
            setSupportZoom(false)
            javaScriptEnabled = true
            domStorageEnabled = true
            allowFileAccess = false
            allowContentAccess = false
        }

        visibility = GONE
    }

    override fun destroy() {
        super.destroy()
        // TODO. Test if it's necessary.
        webViewClient = WebViewClientCompat()
    }

    private val _webViewClient = MraidWebViewClient(context, mraidJsCommandUrlSource).also {
        webViewClient = it
    }

    val isLoaded = _webViewClient.isLoaded
    val unrecoverableError = _webViewClient.unrecoverableError

    // TODO. Refactor. Move to BaseWebView?
    // Currently it's single-time use, so no worries regarding isLoaded flag volatility.
    suspend fun loadHtml(html: String): Result<MraidAdData, MraidAdError> = coroutineScope {
        withContext(DispatcherProvider().main) {
            try {
                loadDataWithDefaultBaseUrl(applyCSSRenderingFix(html))
            } catch (e: Exception) {
                MolocoLogger.error(TAG, e.toString())
                return@withContext Result.Failure(
                    MraidAdError.MRAID_WEBVIEW_LOAD_DATA_WITH_BASE_URL_ERROR
                )
            }

            val (isLoaded, unrecoverableError) =
                _webViewClient.isLoaded
                    .combine(
                        _webViewClient.unrecoverableError
                    ) { isLoaded, unrecoverableError ->
                        isLoaded to unrecoverableError
                    }.first {
                        val (isLoaded, unrecoverableError) = it
                        isLoaded || unrecoverableError != null
                    }

            if (unrecoverableError != null) {
                Result.Failure(unrecoverableError)
            } else if (isLoaded) {
                Result.Success(MraidAdData())
            } else {
                Result.Failure(MraidAdError.MRAID_WEBVIEW_PAGE_DIDNT_LOAD_ERROR)
            }
        }
    }
}

private const val TAG = "MraidWebView"