package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer

import android.webkit.RenderProcessGoneDetail
import android.webkit.WebView
import androidx.annotation.VisibleForTesting
import androidx.webkit.WebViewClientCompat
import com.moloco.sdk.internal.MolocoLogger
import com.moloco.sdk.internal.services.ClickthroughService
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.errors.StaticAdError
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.staticrenderer.model.BannerAdTouch
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ButtonRecorder
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.ButtonTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService
import kotlinx.coroutines.CoroutineScope
import kotlinx.coroutines.flow.MutableSharedFlow
import kotlinx.coroutines.flow.MutableStateFlow
import kotlinx.coroutines.flow.SharedFlow
import kotlinx.coroutines.flow.StateFlow
import kotlinx.coroutines.flow.asStateFlow
import kotlinx.coroutines.launch

internal class WebViewClientImpl(
    private val scope: CoroutineScope,
    private val clickthroughService: ClickthroughService,
    val buttonTracker: ButtonTracker = ButtonTracker()
) : WebViewClientCompat(), ButtonRecorder {

    private val _isLoaded = MutableStateFlow(false)
    val isLoaded: StateFlow<Boolean> = _isLoaded

    private val _isPageFinished = MutableStateFlow(false)
    val isPageFinished = _isPageFinished.asStateFlow()

    /**
     * Workaround: call for cases when some of the web page resources (_img_ tag for example) take too much time to load:
     * [onPageFinished] not called in a while - the place where we consider a web page/ad loaded.
     *
     * This hurts Moloco's static ad fill-rate: lots of ad load timeouts.
     *
     * Until we figure out why this happens,
     * we'll force-enable [isLoaded] flag after some predefined delay, or [onPageFinished] gets there first.
     */
    fun enforceIsLoaded() {
        _isLoaded.value = true
    }

    private val _unrecoverableError = MutableStateFlow<StaticAdError?>(null)
    val unrecoverableError = _unrecoverableError.asStateFlow()

    // TODO. Why not just a simple listener? Auto cancellation of late events is cool though.
    private val _clickthroughEvent = MutableSharedFlow<Unit>()
    val clickthroughEvent: SharedFlow<Unit> = _clickthroughEvent

    @VisibleForTesting
    var lastTouch: BannerAdTouch? = null
        private set

    fun onLastTouch(bannerAdTouch: BannerAdTouch) {
        lastTouch = bannerAdTouch
    }

    override fun onButtonRendered(button: CustomUserEventBuilderService.UserInteraction.Button) {
        buttonTracker.onButtonRendered(button)
    }

    override fun onButtonUnRendered(
        buttonType: CustomUserEventBuilderService.UserInteraction.Button.ButtonType
    ) {
        buttonTracker.onButtonUnRendered(buttonType)
    }

    // Old version of the method works for api 33 as well, so I leave it as is.
    // This annotation is required by Kotlin in order to avoid warnings.
    @Deprecated("Deprecated in Android API 24")
    override fun shouldOverrideUrlLoading(view: WebView?, url: String?): Boolean {
        url?.let {
            lastTouch?.let { touch ->
                scope.launch {
                    clickthroughService.runBannerClickThrough(
                        url,
                        touch,
                        buttonTracker,
                        _clickthroughEvent
                    )
                }
            }
        }
        // Stop loading the url in the webview.
        return true
    }

    override fun onPageFinished(view: WebView?, url: String?) {
        super.onPageFinished(view, url)
        _isLoaded.value = true
        _isPageFinished.value = true
    }

    /**
     * Report an error to the host application. These errors are unrecoverable (i.e. the main resource is unavailable). The errorCode parameter corresponds to one of the ERROR_* constants.
     */
    // Looking for unrecoverable errors only, hence deprecated function is better for that (?)
    // This annotation is required by Kotlin in order to avoid warnings.
    @Suppress("DEPRECATION")
    @Deprecated("Deprecated in Android API 23")
    override fun onReceivedError(
        view: WebView?,
        errorCode: Int,
        description: String?,
        failingUrl: String?
    ) {
        super.onReceivedError(view, errorCode, description, failingUrl)
        _unrecoverableError.value = StaticAdError.STATIC_AD_WEBVIEW_RECEIVED_ERROR
        MolocoLogger.error(TAG, "onReceivedError $description")
    }

    override fun onRenderProcessGone(view: WebView?, detail: RenderProcessGoneDetail?): Boolean {
        // TODO. Logging.
        // https://developer.android.com/guide/webapps/managing-webview#termination-handle
        // Basically, then webview will be destroyed externally after this, which, ideally, isn't known here.
        // But who cares, plus deadlines.
        _unrecoverableError.value = StaticAdError.STATIC_AD_WEBVIEW_RENDER_PROCESS_GONE_ERROR
        MolocoLogger.error(TAG, "onRenderProcessGone")
        return true
    }

    companion object{
        private const val TAG = "WebViewClientImpl"
    }
}
