package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast

import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Companion
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.CreativeType
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.Icon
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.MediaFile
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.model.VastResource
import kotlin.math.absoluteValue

internal fun getVastResourceComparatorBestFirst() = VastResourceComparatorBestFirst

internal fun getCompanionComparatorBestFirst(
    targetWidthPx: Int?,
    targetHeightPx: Int?
): Comparator<Companion> = CompanionComparatorBestFirst(
    targetWidthPx,
    targetHeightPx
)

internal fun getIconComparatorBestFirst() = IconComparatorBestFirst

internal fun getMediaFileComparatorBestFirst(
    targetFileSizeInMegabytes: Double,
    adDurationMillis: Long?,
    targetWidthPx: Int?,
    targetHeightPx: Int?,
): Comparator<MediaFile> = MediaFileComparatorBestFirst(
    targetFileSizeInMegabytes,
    adDurationMillis,
    targetWidthPx,
    targetHeightPx
)

private val VastResourceComparatorBestFirst = Comparator<VastResource> { vr1, vr2 ->
    // vr2 here for reversed/biggest cmpValue to be first.
    vr2.cmpValue.compareTo(vr1.cmpValue)
}

private val IconComparatorBestFirst = Comparator<Icon> { i1, i2 ->
    VastResourceComparatorBestFirst.compare(i1.resource, i2.resource)
}

private class CompanionComparatorBestFirst(
    private val targetWidthPx: Int? = null,
    private val targetHeightPx: Int? = null,
) : Comparator<Companion> {

    override fun compare(c1: Companion, c2: Companion): Int {
        val (c1CmpVal, c2CmpVal) = compareCompanions(
            c1,
            c2,
            targetWidthPx,
            targetHeightPx
        )
        // c2 here for reversed/biggest cmpValue to be first.
        return c2CmpVal.compareTo(c1CmpVal)
    }
}

// TODO. Duplicate of compareMediaFiles.
private fun compareCompanions(
    c1: Companion,
    c2: Companion,
    targetWidthPx: Int? = null,
    targetHeightPx: Int? = null
): Pair<Int, Int> {
    var c1Score = 0
    var c2Score = 0

    // TODO. Duplicate.
    val incScoreIfWon: (Pair<Boolean, Boolean>) -> Unit = { (isFirstWon, isSecondWon) ->
        if (isFirstWon) c1Score++
        if (isSecondWon) c2Score++
    }

    // Can be tweaked. Preliminary version.

    compareByArea(c1, c2, targetWidthPx, targetHeightPx)
        .also(incScoreIfWon)

    compareByAspectRatio(c1, c2, targetWidthPx, targetHeightPx)
        .also(incScoreIfWon)

    compareByAvailableVastResources(c1, c2)
        .also(incScoreIfWon)

    return c1Score to c2Score
}

private class MediaFileComparatorBestFirst(
    private val targetFileSizeInMegabytes: Double,
    private val adDurationMillis: Long? = null,
    private val targetWidthPx: Int? = null,
    private val targetHeightPx: Int? = null,
) : Comparator<MediaFile> {

    override fun compare(mf1: MediaFile, mf2: MediaFile): Int {
        val (mf1CmpVal, mf2CmpVal) = compareMediaFiles(
            mf1,
            mf2,
            targetFileSizeInMegabytes,
            adDurationMillis,
            targetWidthPx,
            targetHeightPx
        )
        // mf2 here for reversed/biggest cmpValue to be first.
        return mf2CmpVal.compareTo(mf1CmpVal)
    }
}

private fun compareMediaFiles(
    mf1: MediaFile,
    mf2: MediaFile,
    targetFileSizeInMegabytes: Double,
    adDurationMillis: Long? = null,
    targetWidthPx: Int? = null,
    targetHeightPx: Int? = null
): Pair<Int, Int> {
    var mf1Score = 0
    var mf2Score = 0

    // TODO. Duplicate.
    val incScoreIfWon: (Pair<Boolean, Boolean>) -> Unit = { (isFirstWon, isSecondWon) ->
        if (isFirstWon) mf1Score++
        if (isSecondWon) mf2Score++
    }

    // Can be tweaked. Preliminary version.

    compareByArea(mf1, mf2, targetWidthPx, targetHeightPx)
        .also(incScoreIfWon)

    compareByAspectRatio(mf1, mf2, targetWidthPx, targetHeightPx)
        .also(incScoreIfWon)

    compareByFileSize(mf1, mf2, targetFileSizeInMegabytes, adDurationMillis)
        .also(incScoreIfWon)

    return mf1Score to mf2Score
}

private fun compareByAvailableVastResources(c1: Companion, c2: Companion): Pair<Boolean, Boolean> {
    val hasStaticResource: (VastResource) -> Boolean = { it is VastResource.Static }

    val c1HasStaticResource = c1.resources.firstOrNull(hasStaticResource) != null
    val c2HasStaticResource = c2.resources.firstOrNull(hasStaticResource) != null

    return c1HasStaticResource to c2HasStaticResource
}

private fun compareByArea(
    c1: Companion,
    c2: Companion,
    targetWidthPx: Int?,
    targetHeightPx: Int?
): Pair<Boolean, Boolean> = commonCoefficientComparator(
    calculateAreaDeviation(c1.widthPx, c1.heightPx, targetWidthPx, targetHeightPx),
    calculateAreaDeviation(c2.widthPx, c2.heightPx, targetWidthPx, targetHeightPx)
)

private fun compareByArea(
    mf1: MediaFile,
    mf2: MediaFile,
    targetWidthPx: Int?,
    targetHeightPx: Int?
): Pair<Boolean, Boolean> = commonCoefficientComparator(
    calculateAreaDeviation(mf1.widthPx, mf1.heightPx, targetWidthPx, targetHeightPx),
    calculateAreaDeviation(mf2.widthPx, mf2.heightPx, targetWidthPx, targetHeightPx)
)

private fun compareByAspectRatio(
    c1: Companion,
    c2: Companion,
    targetWidthPx: Int?,
    targetHeightPx: Int?
): Pair<Boolean, Boolean> = commonCoefficientComparator(
    calculateAspectRatioDeviation(c1.widthPx, c1.heightPx, targetWidthPx, targetHeightPx),
    calculateAspectRatioDeviation(c2.widthPx, c2.heightPx, targetWidthPx, targetHeightPx)
)

private fun compareByAspectRatio(
    mf1: MediaFile,
    mf2: MediaFile,
    targetWidthPx: Int?,
    targetHeightPx: Int?
): Pair<Boolean, Boolean> = commonCoefficientComparator(
    calculateAspectRatioDeviation(mf1.widthPx, mf1.heightPx, targetWidthPx, targetHeightPx),
    calculateAspectRatioDeviation(mf2.widthPx, mf2.heightPx, targetWidthPx, targetHeightPx)
)

private fun compareByFileSize(
    mf1: MediaFile,
    mf2: MediaFile,
    targetFileSizeInMegabytes: Double,
    adDurationMillis: Long?
): Pair<Boolean, Boolean> = commonCoefficientComparator(
    calculateFileSizeDeviation(mf1, targetFileSizeInMegabytes, adDurationMillis),
    calculateFileSizeDeviation(mf2, targetFileSizeInMegabytes, adDurationMillis)
)

// TODO. Rename
private fun commonCoefficientComparator(
    mf1Coef: Double?,
    mf2Coef: Double?
): Pair<Boolean, Boolean> {
    // If nothing is available, well, it's equally bad...
    // If only one Coefficient is missing, give the win to the existing one.
    if (mf1Coef == null || mf2Coef == null) {
        return (mf1Coef != null) to (mf2Coef != null)
    }

    // Equally distant from the target.
    if (mf1Coef == mf2Coef) {
        return true to true
    }

    // Equally distant but positive vs negative Coefficient: I prefer the smaller one.
    if (mf1Coef.absoluteValue == mf2Coef.absoluteValue) {
        return (mf1Coef < 0) to (mf2Coef < 0)
    }

    // Coefficient which is closer to zero, also is closer to the target.
    // Smaller is better here.
    return if (mf1Coef.absoluteValue.compareTo(mf2Coef.absoluteValue) < 0) {
        true to false
    } else {
        false to true
    }
}

/**
 * Basically width x height or whatever is put there. It doesn't really matter.
 * @return
 * Ideal area/No deviation from the target area: 0.0;
 * Area is smaller than the target one: negative value;
 * Area is bigger than the target one: positive value;
 * Not enough/invalid input data to calculate: null;
 */
private fun calculateAreaDeviation(
    widthPx: Int?,
    heightPx: Int?,
    targetWidthPx: Int?,
    targetHeightPx: Int?
): Double? {
    // TODO. Duplication. Standalone method should involve nullability contract which is in beta now.
    if (widthPx == null || widthPx == 0 ||
        heightPx == null || heightPx == 0 ||
        targetWidthPx == null || targetWidthPx == 0 ||
        targetHeightPx == null || targetHeightPx == 0
    ) {
        return null
    }

    val area = widthPx * heightPx.toDouble()
    val targetArea = targetWidthPx * targetHeightPx.toDouble()

    return area - targetArea
}

/**
 * Basically width/height or whatever is put there. It doesn't really matter.
 * @return
 * Ideal aspect ratio/No deviation from the target size: 0.0;
 * Ratio is narrower than the target one: negative value;
 * Ratio is wider than the target one: positive value;
 * Not enough/invalid input data to calculate: null;
 */
private fun calculateAspectRatioDeviation(
    widthPx: Int?,
    heightPx: Int?,
    targetWidthPx: Int?,
    targetHeightPx: Int?
): Double? {
    // TODO. Duplication. Standalone method should involve nullability contract which is in beta now.
    if (widthPx == null || widthPx == 0 ||
        heightPx == null || heightPx == 0 ||
        targetWidthPx == null || targetWidthPx == 0 ||
        targetHeightPx == null || targetHeightPx == 0
    ) {
        return null
    }

    val aspectRatio = widthPx / heightPx.toDouble()
    val targetAspectRatio = targetWidthPx / targetHeightPx.toDouble()

    return aspectRatio - targetAspectRatio
}

/**
 * @return
 * Ideal file size/No deviation from the target size: 0.0;
 * File is smaller than ideal: negative value;
 * File is bigger than ideal: positive value;
 * Not enough data to calculate: null;
 */
private fun calculateFileSizeDeviation(
    mediaFile: MediaFile,
    targetFileSizeInMegabytes: Double,
    adDurationMillis: Long?,
): Double? {
    val fileSizeInMegabytes = calculateFileSizeInMegabytes(mediaFile, adDurationMillis)
        ?: return null

    return fileSizeInMegabytes - targetFileSizeInMegabytes
}

private fun calculateFileSizeInMegabytes(mediaFile: MediaFile, adDurationMillis: Long?): Double? {
    if (adDurationMillis == null) {
        return null
    }

    // KiloBITS per second.
    val bitrateInKbps: Double = with(mediaFile) {
        when {
            bitrate != null -> bitrate.toDouble()
            maxBitrate != null && minBitrate != null -> (maxBitrate - minBitrate) / 2.0
            else -> null
        }
    } ?: return null

    val adDurationSeconds: Double = adDurationMillis / 1000.0

    // Megabytes per second = Kilobits per second ÷ 8,192
    return bitrateInKbps / 8192 * adDurationSeconds
}

private val VastResource.cmpValue: Int
    get() = when (this) {
        is VastResource.Html -> 4
        is VastResource.Static -> if (resource.creativeType == CreativeType.JS) 3 else 2
        is VastResource.IFrame -> 1
    }
