package com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose

import androidx.compose.animation.AnimatedVisibility
import androidx.compose.foundation.layout.Box
import androidx.compose.foundation.layout.BoxScope
import androidx.compose.foundation.layout.PaddingValues
import androidx.compose.foundation.layout.padding
import androidx.compose.runtime.Composable
import androidx.compose.runtime.LaunchedEffect
import androidx.compose.runtime.collectAsState
import androidx.compose.runtime.getValue
import androidx.compose.runtime.mutableStateOf
import androidx.compose.runtime.remember
import androidx.compose.runtime.setValue
import androidx.compose.ui.Alignment
import androidx.compose.ui.Modifier
import androidx.compose.ui.input.pointer.pointerInput
import com.moloco.sdk.internal.scheduling.DispatcherProvider
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ViewVisibilityTracker
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.DEFAULT_AD_BUTTON_PADDING
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.ui.utils.ScreenUtils.toPosition
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PlaybackProgress
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.PreparedVastResource
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.ValueWrapper
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.compose.touch.detectTapUnconsumed
import com.moloco.sdk.xenoss.sdkdevkit.android.adrenderer.internal.vast.render.linear.LinearViewModel
import com.moloco.sdk.xenoss.sdkdevkit.android.core.services.CustomUserEventBuilderService.UserInteraction.Button
import kotlinx.coroutines.withContext

@Composable
internal fun Linear(
    viewModel: LinearViewModel,
    overrideVideoContainerOnClick: (() -> Unit)?,
    modifier: Modifier = Modifier,
    muteButton: MuteButton?,
    progressBar: ProgressBar?,
    vastIcon: VastIcon?,
    playbackControl: PlaybackControl?,
    viewVisibilityTracker: ViewVisibilityTracker,
    onShouldReplay: () -> Unit,
) {
    LaunchedEffect(Unit) {
        withContext(DispatcherProvider().main) {
            viewModel.onDisplayStarted()
        }
    }

    Box(modifier = modifier) {
        val seekToMillis by viewModel.startFromMillis.collectAsState()
        val mute by viewModel.mute.collectAsState()

        var isPlayingLocal by remember {
            mutableStateOf(false)
        }

        var progressLocal by remember {
            mutableStateOf<PlaybackProgress>(
                PlaybackProgress.NotAvailable
            )
        }

        // Autoplay initially.
        var playPauseLocal by remember {
            mutableStateOf(ValueWrapper(true))
        }

        VastVideoPlayer(
            uri = viewModel.videoUri,
            isStreamingEnabled = viewModel.isStreamingEnabled,
            play = playPauseLocal,
            seekToMillis = seekToMillis,
            isMute = mute,
            isPlaying = {
                viewModel.onIsPlaying(it)
                isPlayingLocal = it
            },
            isVisible = {
                if (it) {
                    viewModel.onVideoPlayerVisible()
                } else {
                    viewModel.onVideoPlayerNotVisible()
                }
            },
            viewVisibilityTracker,
            onProgressChanged = {
                viewModel.onProgress(it)
                progressLocal = it
            },
            onError = viewModel::onError,
            modifier = Modifier
                .matchParentSize()
                .pointerInput(Unit) {
                    detectTapUnconsumed { firstTapTouch, _ ->
                        overrideVideoContainerOnClick?.let { it1 -> it1() }
                            ?: viewModel.onClickThrough(firstTapTouch.toPosition())
                    }
                }
        )

        playbackControl?.let {
            val onShouldPlay = { shouldPlay: Boolean ->
                playPauseLocal = ValueWrapper(shouldPlay)
            }

            it(isPlayingLocal, progressLocal, onShouldPlay, onShouldReplay)
        }

        val hasMore = when(progressLocal) {
            is PlaybackProgress.Finished -> false
            else -> true
        }

        // The video player can be playing either when the video is being played or
        // it is being buffered (not actively played) but there is more to be played
        val isPlayerPlaying = isPlayingLocal || hasMore

        muteButton?.let {
            it(isPlayerPlaying, mute, { button: Button, buttonType: Button.ButtonType ->
                viewModel.onButtonRendered(button)
                viewModel.onButtonUnRendered(buttonType)
            }, viewModel::onMuteChange)
        }

        progressBar?.let {
            it(isPlayingLocal, progressLocal)
        }

        vastIcon?.let {
            val vastPrivacyIcon by viewModel.vastPrivacyIcon.collectAsState()
            it(
                vastPrivacyIcon,
                viewModel::onVastPrivacyIconDisplayed,
                viewModel::onVastPrivacyIconClick
            )
        }
    }
}

typealias VastIcon = @Composable BoxScope.(
    icon: PreparedVastResource?,
    onDisplayed: () -> Unit,
    onClick: () -> Unit
) -> Unit

@Composable
fun defaultVastIcon(
    alignment: Alignment = Alignment.BottomStart,
    padding: PaddingValues = PaddingValues(DEFAULT_AD_BUTTON_PADDING)
): VastIcon = { iconResource, onDisplayed, onClick ->
    AnimatedVisibility(
        visible = iconResource != null,
        modifier = Modifier
            .align(alignment)
            .padding(padding)
    ) {
        iconResource?.let {
            VastIcon(
                resource = it,
                onDisplayed = onDisplayed,
                onClick = onClick
            )
        }
    }
}
