/* Copyright 2016 Mountain Fog, Inc.

Licensed under the Apache License, Version 2.0 (the "License");
you may not use this file except in compliance with the License.
You may obtain a copy of the License at

http://www.apache.org/licenses/LICENSE-2.0

Unless required by applicable law or agreed to in writing, software
distributed under the License is distributed on an "AS IS" BASIS,
WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
See the License for the specific language governing permissions and
limitations under the License. */

package com.mtnfog.entity;

import org.apache.commons.lang3.builder.EqualsBuilder;
import org.apache.commons.lang3.builder.HashCodeBuilder;

/**
 * A span in text identified by token and character indexes.
 * 
 * @author Mountain Fog, Inc.
 *
 */
public class Span {

	private int tokenStart;
	private int tokenEnd;
	private int characterStart;
	private int characterEnd;
	
	/**
	 * Creates a new span.
	 * @param tokenStart The position of the start of the token's span.
	 * @param tokenEnd The position of the end of the token's span.
	 * @param characterStart The character-based position of the span.
	 * @param characterEnd The character-based position of the span.
	 * 
	 */
	public Span(int tokenStart, int tokenEnd, int characterStart, int characterEnd) {
		
		this.tokenStart = tokenStart;
		this.tokenEnd = tokenEnd;
		this.characterStart = characterStart;
		this.characterEnd = characterEnd;
		
	}
	
	/**
	 * Returns the token-based span as a formatted string.
	 * An example is: [3..5)
	 */
	@Override
	public String toString() {

		return "[" + tokenStart + ".." + tokenEnd + ")";
		
	}
	
	/**
     * {@inheritDoc}
     */
	@Override
	public int hashCode() {
		
		return new HashCodeBuilder(17, 31)
			.append(tokenStart)
            .append(tokenEnd)
            .append(characterStart)
            .append(characterEnd)
            .toHashCode();
		
	}
	
	/**
     * {@inheritDoc}
     */
	@Override
	public boolean equals(Object obj) {
		
	    if(obj != null && obj instanceof Span) {
	    	
	        final Span other = (Span) obj;
	        
	        return new EqualsBuilder()
	            .append(tokenStart, other.tokenStart)
	            .append(tokenEnd, other.tokenEnd)
	            .append(characterStart, other.characterStart)
	            .append(characterEnd, other.characterEnd)
	            .isEquals();
	        
	    }

	    return false;
	    
	}
	
	/**
	 * Gets the token's start of the span.
	 * @return The token's start of the span.
	 */
	public int getTokenStart() {
		return tokenStart;
	}
	
	/**
	 * Gets the end of the span.
	 * @return The end of the span.
	 */
	public int getTokenEnd() {
		return tokenEnd;
	}

	/**
	 * Gets the character's start of the span.
	 * @return The character's start of the span.
	 */
	public int getCharacterStart() {
		return characterStart;
	}

	/**
	 * Sets the character-based index of the span.
	 * @param characterStart The character-based index of the span.
	 */
	public void setCharacterStart(int characterStart) {
		this.characterStart = characterStart;
	}

	/**
	 * Gets the character's end of the start of the span.
	 * @return The character's end of the start of the span.
	 */
	public int getCharacterEnd() {
		return characterEnd;
	}

	/**
	 * Sets the character-based index of the end of the span.
	 * @param characterStart The character-based index of the end of the span.
	 */
	public void setCharacterEnd(int characterEnd) {
		this.characterEnd = characterEnd;
	}
	
}