package com.mule.connectors.testdata;

import com.mule.connectors.testdata.model.ConnectorProcessorInfo;
import com.mule.connectors.testdata.parsers.AnnotationsXmlParser;
import com.mule.connectors.testdata.parsers.ConnectorSchemaParser;
import com.mule.connectors.testdata.templates.AutomationBeansTemplate;
import com.mule.connectors.testdata.templates.PropertiesOverriderTemplate;
import com.mule.connectors.testdata.parsers.StudioEditorsXmlParser;


import com.mule.connectors.testdata.utils.DocumentHandler;
import org.apache.log4j.Level;
import org.apache.maven.plugin.AbstractMojo;
import org.apache.maven.plugin.MojoExecutionException;
import org.apache.maven.plugin.MojoFailureException;

import org.apache.log4j.Logger;

import org.w3c.dom.Element;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPathExpressionException;
import java.io.File;
import java.io.IOException;
import java.util.Map;

/**
 * Echos an object string to the output screen.
 * @goal parse
 * @phase package
 * @requiresProject false
 */
public class TestDataParserMojo extends AbstractMojo
{

    /**
     * Input Schema file, generated with devkit.
     * @parameter expression="${testdata.logLevel}" default-value="info"
     */
    private String logLevel;

    /**
     * Input file containing connector's basic credentials
     * @parameter expression="${testdata.credentialsFile}" default-value=" "
     */
    private String credentialsFile;

    /**
     *  Input xml file, generated with the testdata annotations processor,
     * contains all the properties from the connector and its processors.
     *
     * @parameter expression="${testdata.annotationsFile}" default-value=" "
     */
    private String annotationsFile;

    /**
     * Output generated xml file
     * @parameter expression="${testdata.outputFile}" default-value="output.xml"
     */
    private String outputFile;

    /**
     * Target directory present in root build directory
     * @parameter default-value="${project.build.directory}"
     * */
    private String targetDir;

    /**
     * Build root directory
     * @parameter expression="${testdata.outputDir}" default-value="${project.basedir}/src/test/resources"
     * */
    private String outputDir;


    private File schemaFile = null;
    private File editorsFile = null;

    public void execute()
            throws MojoExecutionException, MojoFailureException
    {
        try{
            if(getLog().isDebugEnabled() || logLevel.equals("debug"))
                Logger.getRootLogger().setLevel(Level.DEBUG);

            outputFile = (outputFile.equals("") ? "output.xml": outputFile);
            outputDir = (outputFile.equals("") ? "./src/test/resources" : outputDir);
            DocumentHandler.setOutputPath(outputDir);

            getLog().info("=================================================");
            getLog().info("******** Begin Parsing of TestData files ********");
            getLog().info("=================================================");

            findEditorsAndSchemaFiles();

            getLog().info("** Parsing Schema file ");

            ConnectorSchemaParser connectorSchemaParser =
                    new ConnectorSchemaParser(schemaFile, outputFile);

            connectorSchemaParser.parseComplexTypeNodesDefinitions();
            connectorSchemaParser.parseRootElementNodesDefinitions();

            getLog().info("** Parsing Editors file ");
            StudioEditorsXmlParser studioEditorsXmlParser = new StudioEditorsXmlParser(editorsFile);

            studioEditorsXmlParser.addStudioAttributePropertiesToProcessors(connectorSchemaParser.getProcessorsElement());
            studioEditorsXmlParser.addStudioAttributePropertiesToGlobalConfig(connectorSchemaParser.getConfigElement());

            getLog().info("** Parsing Annotations Processor output file ");
            AnnotationsXmlParser annotationsXMLParser = new AnnotationsXmlParser(annotationsFile);
            Element connectorsPropertiesNode = annotationsXMLParser.getPropertiesNode();
            Map<String, ConnectorProcessorInfo> processorsProperties = annotationsXMLParser.retrieveProcessorsProperties();

            getLog().info("** Creating Overrides template ");
            PropertiesOverriderTemplate overridesTemplate =
                    new PropertiesOverriderTemplate(connectorSchemaParser.getConnectorName(), outputFile);

            overridesTemplate.setConfigElement(connectorSchemaParser.getConfigElement(), credentialsFile);
            overridesTemplate.setMessageProcessorsElement(connectorSchemaParser.getProcessorsElement(), processorsProperties);
            overridesTemplate.setPropertiesElement(connectorsPropertiesNode);

            getLog().info("** Creating Automation Beans template");
            AutomationBeansTemplate automationBeansTemplate = new AutomationBeansTemplate();
            automationBeansTemplate.createAutomationBeansFromProcessors(connectorSchemaParser.getMessageProcessors());

            getLog().info("** Export Basic Testdata file: " + outputFile);
            connectorSchemaParser.exportToFile();

            getLog().info("** Export Overrides Testdata file");
            overridesTemplate.exportToFile();

            getLog().info("** Export Automation Spring Beans");
            automationBeansTemplate.exportToFile();

            getLog().info("=================================================");
            getLog().info("=================================================");

        } catch (XPathExpressionException e) {
            getLog().error(e.toString());
            throw new RuntimeException(e);

        } catch (ParserConfigurationException e) {
            getLog().error(e.toString());
            throw new RuntimeException(e);

        } catch (SAXException e) {
            getLog().error(e.toString());
            throw new RuntimeException(e);

        } catch (IOException e) {
            getLog().error(e.toString());
            throw new RuntimeException(e);

        } catch (TransformerConfigurationException e) {
            getLog().error(e.toString());
            throw new RuntimeException(e);

        } catch (TransformerException e) {
            getLog().error(e.toString());
            throw new RuntimeException(e);

        }
    }


    private void findEditorsAndSchemaFiles() {
        getLog().debug("*** Path to META-INF :: " + targetDir +"/generated-sources/mule/META-INF");
        String xsdName = "";

        File metaInf = new File(targetDir +"/generated-sources/mule/META-INF");
        for(File temp: metaInf.listFiles()){
            if(temp.isFile()){
                getLog().debug("*** File in META :: " + temp.getName());
                if(temp.getName().contains(".xsd")){
                    xsdName=temp.getName();

                    schemaFile = temp;
                }
            }
        }

        String editorsDirectory = xsdName.replace("mule-","").replace(".xsd","");
        File mule = new File(targetDir +"/generated-sources/mule");
        for(File temp: mule.listFiles()){
            if(temp.isDirectory() && temp.getName().equals(editorsDirectory))
                for(File inner: temp.listFiles()){
                    getLog().debug("*** File in mule :: " + inner.getName());

                    if(inner.getName().equals("editors.xml"))
                        editorsFile = inner;
                }
        }
    }

}
