package com.mule.connectors.testdata.parsers;


import com.mule.connectors.testdata.model.naming.GeneralAttributes;
import com.mule.connectors.testdata.model.naming.OutputTagNames;
import com.mule.connectors.testdata.model.naming.SchemaTagNames;
import com.mule.connectors.testdata.utils.DocumentHandler;
import com.mule.connectors.testdata.utils.XPathEvaluator;
import org.apache.log4j.Logger;

import org.w3c.dom.Element;
import org.w3c.dom.NodeList;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.soap.Node;
import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPathExpressionException;
import java.io.File;
import java.io.IOException;
import java.util.*;

public class ConnectorSchemaParser {

    private Logger logger = Logger.getLogger(ConnectorSchemaParser.class);
    private final String CACHEDTYPE_ID = "cachedTypeId";

    /**
     * Input Schema file .
     */
    private DocumentHandler sourceDoc;

    /**
     * Output generated xml file
     */
    private String outputFile;
    private DocumentHandler outputDoc;

    /**
     * Map with (ComponentName, ComponentType) relations.
     * The value for each key its a reference to the type root Element
     */
    private Map<String, Element> sourceTypes = new HashMap<String, Element>();

    private Map<String, Map<String, List<Element>>> baseTypesCache = new HashMap<String, Map<String, List<Element>>>();


    /**
     * Set of connection configuration keys for fast lookup of attributes.
     * */
    private Set<String> configKeys = new HashSet<String>();


    private String connectorName;

    private XPathEvaluator xPathEvaluator;

    private Element configElement = null;
    private Element processorsElement = null;
    private Element inboundEndpointElement = null;
    private Element outboundEndpointElement = null;

    public Element getMessageProcessors(){ return processorsElement;}


    public ConnectorSchemaParser(File inputFile, String outputFile)
            throws IOException, SAXException, ParserConfigurationException
    {

        logger.debug("- Initialize SchemaParser");
        this.outputFile = outputFile;

        logger.debug("  Reading input file: " + inputFile);

        sourceDoc = new DocumentHandler(inputFile);
        outputDoc = new DocumentHandler();

        xPathEvaluator = new XPathEvaluator(this.sourceDoc.getDoc());

        connectorName = inputFile.getCanonicalPath().substring(inputFile.getCanonicalPath().lastIndexOf('/') + 1)
                .replace(".xsd", "")
                .replace("mule-", "");
    }


    /**
     *  Parses all the types definitions found in the schema file,
     *  creating a lookup table for the elements by name.
     *
     * */
    public void parseComplexTypeNodesDefinitions() throws XPathExpressionException {

        logger.debug("- Parse complex type definitions");
        // Retrieve all complex types
        String typeElementsXPathExpression = "//xs:complexType";
        NodeList typeNodes = xPathEvaluator.evaluateOnDocAndGetNodeList(typeElementsXPathExpression);

        logger.debug("  Retrived nodes");
        // Generate mapping with pairs (element, elementTypeDefinition)
        sourceTypes = parseTypeNodesByName(typeNodes);
    }

    /**
     *   Parses all the top level elements found in the schema file,
     *  from the root.
     *   Retrieved Elements are appendend in four categories, creating the
     *  output document structure.
     *
     * */
    public void parseRootElementNodesDefinitions() throws XPathExpressionException {

        logger.debug("- Parse root element definitions");

        // Initialize root element
        Element rootOutElem = outputDoc.createRootElement(OutputTagNames.CONNECTOR);
        rootOutElem.setAttribute(GeneralAttributes.NAME, connectorName);

        // Main groups in output file
        configElement = outputDoc.createElement(OutputTagNames.GLOBAL_CONFIG);
        inboundEndpointElement = outputDoc.createElement(OutputTagNames.INBOUND_ENDPOINTS);
        outboundEndpointElement = outputDoc.createElement(OutputTagNames.OUTBOUND_ENDPOINTS);
        processorsElement = outputDoc.createElement(OutputTagNames.MESSAGE_PROCESSORS_NODE);

        // Retrieve elements definitions
        String sourceElementsXPathExpression = "/xs:schema/xs:element";
        NodeList sourceElements = xPathEvaluator.evaluateOnDocAndGetNodeList(sourceElementsXPathExpression);

        for (int i=0; i < sourceElements.getLength(); i++){
            Element sourceElem = (Element) sourceElements.item(i);

            // Create node based on element's type definition parsed previously
            Element outputElem = createOutputElementFromDefinition(sourceElem);
            if( outputElem != null ){

                logger.debug("-- Create node based on element's type :: " + outputElem.getTagName());

                if ( sourceElem.getAttribute(GeneralAttributes.SUBSTITUTION_GROUP).contains(GeneralAttributes.MESSAGE_PROCESSOR) ){
                    logger.debug("-- Created Processor Element :: " + outputElem.getTagName());
                    processorsElement.appendChild(outputElem);

                }else if ( sourceElem.getAttribute(GeneralAttributes.SUBSTITUTION_GROUP).contains(GeneralAttributes.INBOUND_ENDPOINTS) ){
                    logger.debug("-- Created Inbound Endpoint Element :: " + outputElem.getTagName());
                    inboundEndpointElement.appendChild(outputElem);

                }else if ( sourceElem.getAttribute(GeneralAttributes.SUBSTITUTION_GROUP).contains(GeneralAttributes.OUTBOUND_ENDPOINTS) ){
                    logger.debug("-- Created Outbound Endpoint Element :: " + outputElem.getTagName());
                    outboundEndpointElement.appendChild(outputElem);

                }else {
                    configElement.appendChild(outputElem);
                }
            }
        }

        rootOutElem.appendChild(configElement);
        rootOutElem.appendChild(processorsElement);
        rootOutElem.appendChild(inboundEndpointElement);
        rootOutElem.appendChild(outboundEndpointElement);
    }

    private void prettyprint(Element e, int indent){

        NodeList top = e.getChildNodes();
        for(int node = 0; node<top.getLength(); node++){
            if(top.item(node).getNodeType() == Node.ELEMENT_NODE){
                Element inner = (Element)top.item(node);
                String ind = "";
                for(int space =0; space<indent; space++) ind += " ";

                logger.debug(ind + "TAG:: " + inner.getTagName() + "   NAME  :: " + inner.getAttribute(GeneralAttributes.NAME));
                prettyprint(inner, indent+1);
            }
        }

    }

    /**
     *  Exports the output document to an xml file
     * */
    public void exportToFile() throws TransformerException {

        logger.debug("- Export file: " + outputFile);

        this.outputDoc.exportToFile(outputFile);
    }


    /**
     *   Given a root 'ComplexContent' node element, parses al the attributes and
     *  childElements nodes appended to the root.
     *
     *   @return a Map with Attributes and ChildElements lists of elements.
     *
     * */
    private Map<String, List<Element>> parseComplexContent(Element complexContentRoot, boolean avoidDuplicates)
            throws XPathExpressionException
    {
        logger.debug("---  Parse complex content Root :: " + complexContentRoot.getTagName());
        logger.debug("---  Parse complex content ID :: " + complexContentRoot.getAttribute(CACHEDTYPE_ID));

        logger.debug("  Retrieve base element nodes");
        Map<String, List<Element>>innerNodes = retrieveBaseElementsNodes(complexContentRoot);

        logger.debug("  Get attributes nodes");
        List<Element> attributes = getElementAttributes(complexContentRoot, avoidDuplicates);
        innerNodes.get(OutputTagNames.ATTRIBUTES_NODE).addAll(attributes);

        logger.debug("  Get child elements");
        Element sequenceNode = (Element)complexContentRoot.getElementsByTagName(SchemaTagNames.SEQUENCE).item(0);
        List<Element> childElements = parseChildElements(sequenceNode);
        innerNodes.get(OutputTagNames.CHILD_ELEMENTS_NODE).addAll(childElements);

        return innerNodes;
    }


    /**
     *   Retrieves all the hierarchy info of a given element, appending found elements
     *  to the innerNodes of the root.
     *
     * */
    private Map<String, List<Element>> retrieveBaseElementsNodes(Element complexContentRoot)
            throws XPathExpressionException
    {
        logger.debug("- Retrieve Base Elements Nodes");
        Map<String, List<Element>> innerNodes = new HashMap<String, List<Element>>();
        innerNodes.put(OutputTagNames.ATTRIBUTES_NODE, new LinkedList<Element>());
        innerNodes.put(OutputTagNames.CHILD_ELEMENTS_NODE, new LinkedList<Element>());

        String expression = "./"+SchemaTagNames.EXTENSION;
        NodeList extensionNodes = xPathEvaluator.evaluateOnElementAndGetNodeList(complexContentRoot, expression);
        for(int i=0; i<extensionNodes.getLength(); i++){

            Element extensionNode = (Element) extensionNodes.item(i);
            Map<String, List<Element>> inheritedNodes = evaluateBase(extensionNode);
            if(inheritedNodes != null){
                logger.debug("-- Obtained from evaluateBase " + inheritedNodes.get("attributes").size() + " attributes");
                String elementName = getParentElementName(extensionNode);
                addInheretedNodesToMyInnerNodes(innerNodes, inheritedNodes, elementName);
            }
        }

        logger.debug("-- Obtained " + innerNodes.get(OutputTagNames.ATTRIBUTES_NODE).size() + " from inner.ATTRIBUTES_NODE");
        logger.debug("-- Obtained " + innerNodes.get(OutputTagNames.CHILD_ELEMENTS_NODE).size() + " from inner.CHILD_ELEMENTS_NODE");
        return innerNodes;
    }

    private Map<String, List<Element>> evaluateBase(Element source)
            throws XPathExpressionException
    {
        String baseType = source.getAttribute(GeneralAttributes.BASE);

        logger.debug("- Evaluate Base :: " + baseType);
        logger.debug("- Source Tag :: " + source.getTagName());

        // If already been parsed, the base type is cached
        if(baseTypesCache.containsKey(baseType)){
            logger.debug("-- Return Cached BaseType :: " + baseType);
            return baseTypesCache.get(baseType);
        }

        logger.debug("-- Parse New BaseType");
        // The first time it is calculated and then cached
        Map<String, List<Element>> baseNodes = null;

        Element baseComplexNode = sourceTypes.get(baseType);
        if (baseComplexNode != null){
            Element complexContent = getComplexContentNode(baseComplexNode);
            if (complexContent != null)
                logger.debug("---- complexContent Id :: " + complexContent.getAttribute(CACHEDTYPE_ID));

            baseNodes = parseComplexContent( complexContent, false);

            if(baseNodes != null){
                logger.debug("---- Save BaseType :: " + baseType);
                baseTypesCache.put(baseType, baseNodes);
            }
        }

        return baseNodes;
    }


    private void addInheretedNodesToMyInnerNodes(Map<String, List<Element>> innerNodes,
                                                 Map<String, List<Element>> inheritedNodes, String elementName)
    {
        logger.debug("- Add inhereted nodes TO :: " + elementName);
        logger.debug("-- innerNodes :: " + innerNodes);
        logger.debug("-- inheritedNodes :: " + inheritedNodes);

        addInheretedAttributesToInnerNodes(innerNodes, inheritedNodes, elementName);

        addInheretedChildsToInnerNodes(innerNodes, inheritedNodes, elementName);

        logger.debug("--- Inner Attributes result :: "+ innerNodes.get(OutputTagNames.ATTRIBUTES_NODE).size());
        logger.debug("--- Inner Childs result :: " + innerNodes.get(OutputTagNames.CHILD_ELEMENTS_NODE).size());

    }

    private void addInheretedAttributesToInnerNodes(Map<String, List<Element>> innerNodes, Map<String, List<Element>> inheritedNodes,
                                                    String elementName)
    {
        logger.debug("-- Add inhereted ATTRIBUTES");

        for(Element inherited: inheritedNodes.get(OutputTagNames.ATTRIBUTES_NODE)){
            if(inherited.getAttribute(GeneralAttributes.BASE).equals(""))
                inherited.setAttribute(GeneralAttributes.BASE, elementName);

            innerNodes.get(OutputTagNames.ATTRIBUTES_NODE).add(inherited);
            logger.debug("--- Add :: " + inherited.getAttribute(GeneralAttributes.NAME) + " ::  TO " + elementName);

        }
    }

    private void addInheretedChildsToInnerNodes(Map<String, List<Element>> innerNodes, Map<String, List<Element>> inheritedNodes,
                                                String elementName)
    {
        logger.debug("-- Add inhereted CHILDS");

        for(Element inherited: inheritedNodes.get(OutputTagNames.CHILD_ELEMENTS_NODE)){
            if(inherited.getAttribute(GeneralAttributes.BASE).equals(""))
                inherited.setAttribute(GeneralAttributes.BASE, elementName);

            innerNodes.get(OutputTagNames.CHILD_ELEMENTS_NODE).add(inherited);
            logger.debug("--- Add :: " + inherited.getAttribute(GeneralAttributes.NAME) + " ::  TO " + elementName);
        }
    }


    private List<Element> getElementAttributes(Element parent,  boolean saveConfigKeys)
            throws XPathExpressionException
    {
        logger.debug("- Get Element Attributes :: " + parent.getTagName());
        List<Element> attributes = new LinkedList<Element>();
        NodeList attributeElems = parent.getElementsByTagName(SchemaTagNames.ATTRIBUTE);

        logger.debug("-- Has " + attributeElems.getLength() + " Attributes");

        for(int j=0; j < attributeElems.getLength(); j++){
            Element srcAttr = (Element) attributeElems.item(j);
            Element newAttr = this.createOutputAttributeElement(srcAttr, !saveConfigKeys);

            if (newAttr != null){
                logger.debug("--- Add Attribute :: " + newAttr.getAttribute(GeneralAttributes.NAME));

                attributes.add(newAttr);

                if( saveConfigKeys )
                    configKeys.add(newAttr.getAttribute(GeneralAttributes.NAME));
            }
        }

        return attributes;
    }


    private String getParentElementName(Element attributesRoot) {

        logger.debug("- Get Parent Element Name" );

        Element n = (Element)attributesRoot.getParentNode();
        while(!n.getNodeName().equals("xs:element") && !n.getNodeName().equals("xs:schema"))
            n = (Element)n.getParentNode();

        logger.debug("-- Parent element name :: " + n.getAttribute(GeneralAttributes.NAME));

        return n.getAttribute(GeneralAttributes.NAME);
    }




    private Element createOutputElementFromDefinition(Element sourceElem)
            throws XPathExpressionException
    {
        logger.debug("- Create outputElement :: " + sourceElem.getAttribute(GeneralAttributes.NAME));

        Element outputElem = this.outputDoc.createElement(sourceElem.getAttribute(GeneralAttributes.NAME));

        Element complexContent = getComplexContentNode(sourceElem);
        if (complexContent != null){

            logger.debug("----  outputElement Name :: " + sourceElem.getAttribute(GeneralAttributes.NAME));

            boolean avoidDuplicates = sourceElem.getAttribute(GeneralAttributes.NAME).contains("config");
            Map<String, List<Element>> innerNodes = parseComplexContent(complexContent, avoidDuplicates);

            appendAttributesFromInnerNodes(outputElem, innerNodes);

            appendChildElementsFromInnerNodes(outputElem, innerNodes);

            return outputElem;
        }

        return null;
    }

    private void appendChildElementsFromInnerNodes(Element outputElem, Map<String, List<Element>> innerNodes) {
        if ( !innerNodes.get(OutputTagNames.CHILD_ELEMENTS_NODE).isEmpty()){
            Element childsNode = evaluateInnerChildElements(innerNodes);

            outputElem.appendChild(childsNode);
        }
    }

    private void appendAttributesFromInnerNodes(Element outputElem, Map<String, List<Element>> innerNodes) {
        if( !innerNodes.get(OutputTagNames.ATTRIBUTES_NODE).isEmpty()){

            Element attributesNode = this.outputDoc.createElement(OutputTagNames.ATTRIBUTES_NODE);

            for(Element attr: innerNodes.get(OutputTagNames.ATTRIBUTES_NODE)){
                Element newAttr = (Element) this.outputDoc.importNode(attr, true);
                attributesNode.appendChild(newAttr);
            }

            outputElem.appendChild(attributesNode);
        }
    }


    private Element createChildElement(Element child) throws XPathExpressionException {

        logger.debug("- Create child Element");

        Element outputElem = this.outputDoc.createElement(getCorrespondingTag(child));
        logger.debug("-- Child Element Name :: " + child.getAttribute(GeneralAttributes.NAME));

        outputElem.setAttribute(GeneralAttributes.NAME, child.getAttribute(GeneralAttributes.NAME));

        Element complexContent = getComplexContentNode(child);
        if (complexContent != null){
            Map<String, List<Element>> innerNodes = parseComplexContent(complexContent, true);

            if( !innerNodes.get(OutputTagNames.ATTRIBUTES_NODE).isEmpty()){
                logger.debug("--- Child Element Append Inner Attributes");

                Element attributesNode = getAttributesOfChildElementFromInnerNodes(innerNodes);
                outputElem.appendChild(attributesNode);
            }

            Element refAttributes = createDefinitionByReferenceAttributeFields();
            outputElem.appendChild(refAttributes);

            if ( !innerNodes.get(OutputTagNames.CHILD_ELEMENTS_NODE).isEmpty()){
                logger.debug("--- Child Element Append ChildElements");

                Element childsNode = evaluateInnerChildElements(innerNodes);
                outputElem.appendChild(childsNode);
            }

            return outputElem;
        }

        return null;
    }

    private String getCorrespondingTag(Element child) {

        boolean isOptional = child.getAttribute(GeneralAttributes.MIN_OCCURS).equals("0");
        return ((isOptional) ? OutputTagNames.OPTIONAL : OutputTagNames.REQUIRED);
    }


    private Element evaluateInnerChildElements(Map<String, List<Element>> innerNodes) {

        logger.debug("- Append Inner ChildElements");

        Element childsNode= this.outputDoc.createElement(OutputTagNames.CHILD_ELEMENTS_NODE);

        Set<Element> childElements = new HashSet<Element>(innerNodes.get(OutputTagNames.CHILD_ELEMENTS_NODE));
        for(Element elem: childElements){
            Element newChild = (Element) this.outputDoc.importNode(elem, true);
            childsNode.appendChild(newChild);
        }

        return childsNode;
    }


    private Element getAttributesOfChildElementFromInnerNodes(Map<String, List<Element>> innerNodes) {

        logger.debug("- Append Inner Attributes");

        Element attributesNode = this.outputDoc.createElement(OutputTagNames.ATTRIBUTES_NODE);
        attributesNode.setAttribute(GeneralAttributes.CONTROLLED, "complex");

        for(Element attr: innerNodes.get(OutputTagNames.ATTRIBUTES_NODE)){
            Element newAttr = (Element) this.outputDoc.importNode(attr, true);
            attributesNode.appendChild(newAttr);
        }

        return attributesNode;
    }


    private Element createDefinitionByReferenceAttributeFields(){

        logger.debug("- Create Definition By ReferenceAttributeFields");

        Element refAttributes = this.outputDoc.createElement(OutputTagNames.ATTRIBUTES_NODE);
        refAttributes.setAttribute(GeneralAttributes.CONTROLLED, "useReference");

        Element refValueAttr = this.outputDoc.createElement("required");
        refValueAttr.setAttribute("name", "ref");
        refAttributes.appendChild(refValueAttr);

        return refAttributes;
    }


    /**
     *  Finds the type definition (complexContent node) of the given element
     *
     *  @return complexContent node if found
     *          null if no definition was found
     * */
    private Element getComplexContentNode(Element parent) throws XPathExpressionException {

        logger.debug("- Get ComplexContentNode");

        Element node = null;
        String expression = "";
        String id;

        if (xPathEvaluator.hasChildElement(parent, SchemaTagNames.COMPLEX_TYPE)){
            // Type definition exists as a child node
            expression = "./xs:complexType/xs:complexContent";
            node = parent;
            id=parent.getAttribute(GeneralAttributes.NAME);

        }else{
            // Type definition exists in a separate node. Searches in the types lookup table
            expression = "./xs:complexContent";

            if (parent.hasAttribute(GeneralAttributes.TYPE))
                id = parent.getAttribute(GeneralAttributes.TYPE);
            else
                id = parent.getAttribute(GeneralAttributes.NAME);

            node = sourceTypes.get(id);
        }


        if(node != null){
            Element contentNode = (Element) xPathEvaluator.evaluateOnElementAndGetNodeList(node, expression).item(0);

            // Definitions exists inside the parent node 'complexType' directly, there's no complexContent
            if(contentNode != null){
                contentNode.setAttribute(CACHEDTYPE_ID, id);
                return contentNode;
            }else{
                node.setAttribute(CACHEDTYPE_ID, id);
                return node;
            }
        }

        logger.debug("-- No ComplexContent definition for :: " + parent.getAttribute(GeneralAttributes.NAME));

        return null;
    }



    private List<Element> parseChildElements(Element sequenceNode)
            throws XPathExpressionException
    {
        logger.debug("- Parse ChildElements");
        List<Element> childElements = new LinkedList<Element>();

        if(sequenceNode != null){
            logger.debug("-- ParseChildElements ::  " + sequenceNode.getTagName());

            NodeList childs = sequenceNode.getElementsByTagName(SchemaTagNames.ELEMENT);
            logger.debug("--- Child Elements :: " + childs.getLength());
            for(int i=0; i<childs.getLength(); i++){
                Element childElement = createChildElement((Element) childs.item(i));

                if(childElement != null)
                    childElements.add(childElement);
            }
        }

        return childElements;
    }




    private Element createOutputAttributeElement(Element sourceNode, boolean avoidDuplicates){

        logger.debug("- Create Output Attribute Element for :: " + sourceNode.getAttribute( GeneralAttributes.NAME ));

        String use = sourceNode.getAttribute( GeneralAttributes.USE ); // required - optional
        String name = sourceNode.getAttribute( GeneralAttributes.NAME );
        String defaultValue = sourceNode.getAttribute( GeneralAttributes.DEFAULT );

        if (name.equals("value-ref") || name.equals("config-ref") ){
            return null;
        }


        if ( avoidDuplicates &&
             use.equals(OutputTagNames.OPTIONAL) && configKeys.contains(name))
        {
            logger.debug("-- Skipped attribute :: " + name);
            // Skips duplicated optional attribute of configuration
            return null;

        }else{
            use = (use.equals("") ? "optional" : use);
            Element attrElem = outputDoc.createElement(use);
            attrElem.setAttribute( GeneralAttributes.NAME , name);

            if( !defaultValue.equals("") )
                attrElem.setAttribute( GeneralAttributes.DEFAULT, defaultValue);

            return attrElem;
        }
    }


    private HashMap<String, Element> parseTypeNodesByName(NodeList typeList){

        logger.debug("- Parse TypeNodes by Name");
        HashMap<String, Element> map = new HashMap<String, Element>();

        for(int i = 0; i<typeList.getLength(); i++){
            Element node = (Element) typeList.item(i);

            if(node.hasAttribute(GeneralAttributes.NAME)){
                logger.debug("-- Parsed TypeNodes :: " + node.getAttribute(GeneralAttributes.NAME));
                map.put(node.getAttribute(GeneralAttributes.NAME), node);
            }
        }

        return map;
    }


    public String getConnectorName(){
        return (this.outputDoc.getDocumentElement().getAttribute(GeneralAttributes.NAME));
    }

    public Element getConfigElement(){
        return configElement;
    }

    public Element getProcessorsElement(){
        return processorsElement;
    }

    public Element getInboundEndpointElement(){
        return inboundEndpointElement;
    }

    public Element getOutboundEndpointElement(){
        return outboundEndpointElement;
    }
}
