package com.mule.connectors.testdata.parsers;

import com.mule.connectors.testdata.model.naming.EditorsTagNames;
import com.mule.connectors.testdata.model.naming.GeneralAttributes;
import com.mule.connectors.testdata.model.naming.OutputTagNames;
import com.mule.connectors.testdata.utils.DocumentHandler;
import com.mule.connectors.testdata.utils.XPathEvaluator;
import org.apache.log4j.Logger;
import org.w3c.dom.*;
import org.xml.sax.SAXException;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;
import java.io.File;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

import static com.mule.connectors.testdata.utils.Utils.copyFieldPropertiesToAttributeNode;

public class StudioEditorsXmlParser {

    private Logger logger = Logger.getLogger(StudioEditorsXmlParser.class);
    private XPathEvaluator xPath;
    private DocumentHandler editorsDoc;

    private Map<String, Element> nestedElements  = new HashMap<String, Element>();
    private Map<String, Element> attributesIndex = new HashMap<String, Element>();

    public StudioEditorsXmlParser(File editorsFile)
            throws ParserConfigurationException, IOException, SAXException, XPathExpressionException
    {
        logger.debug("- Studio Editors XML Parser Initialize");
        this.editorsDoc = new DocumentHandler(editorsFile);
        this.xPath = new XPathEvaluator(editorsDoc.getDoc());

        logger.debug("-- Parse nested elements");
        parseNestedElementsDefinitionNodes(editorsDoc.getDoc());
    }


    private Map<String, Element> parseNestedElementsDefinitionNodes(Document editorsDoc)
            throws XPathExpressionException
    {
        logger.debug("- Parse Nested Elements Definition Nodes");

        NodeList nested = ((Element)editorsDoc.getFirstChild()).getElementsByTagName(EditorsTagNames.NESTED);
        for(int i = 0; i < nested.getLength(); i++){
            Element e = (Element) nested.item(i);
            nestedElements.put(e.getAttribute(GeneralAttributes.LOCAL_ID), e);
        }

        return nestedElements;
    }


    public void addStudioAttributePropertiesToProcessors(Element processorsElementFromSchema)
            throws XPathExpressionException
    {
        logger.debug("- Add Studio Attribute Properties To Processors");

        NodeList connectorElements = (editorsDoc.getDocumentElement().getElementsByTagName(EditorsTagNames.CLOUD_CONNECTOR));
        logger.debug("-- Found " + connectorElements.getLength() + " Connectors");
        for(int i=0; i<connectorElements.getLength(); i++){

            Element cloudElem = (Element) connectorElements.item(i);
            String processorName = cloudElem.getAttribute(GeneralAttributes.LOCAL_ID);

            logger.debug("-- Parsing " +processorName+ " processor");

            Element processor = xPath.getFirstElementWithTag(processorsElementFromSchema, processorName);
            Element attributeCategory = xPath.getFirstElementWithTag(cloudElem, EditorsTagNames.ATTRIBUTE_CATEGORY);

            if( processor != null && attributeCategory != null){
                addStudioPropertiesToProcessorAttributes(processor, attributeCategory);
                parseProcessorChildElement(processor, attributeCategory);
            }
        }
    }

    private void addStudioPropertiesToProcessorAttributes(Element processor, Element attributeCategory)
            throws XPathExpressionException
    {

        logger.debug("- Add Studio Properties To Processors Attributes");

        NodeList processorAttributes = xPath.evaluateOnElementAndGetNodeList(processor, "./attributes/*");
        logger.debug("-- Found " + processorAttributes.getLength() + " processorAttributes");
        for(int p=0; p<processorAttributes.getLength(); p++){

            Element attribute = (Element) processorAttributes.item(p);
            String name = attribute.getAttribute(GeneralAttributes.NAME);

            if( !isConfigAttribute(name)){

                String expression = "./*/*[@name='"+ name +"']";
                Element field = (Element) xPath.evaluateOnElementAndGetNode(attributeCategory,expression);

                if (field != null){

                    copyFieldPropertiesToAttributeNode(attribute, field);

                    Element group = (Element) field.getParentNode();
                    attribute.setAttribute(GeneralAttributes.GROUP, group.getAttribute(GeneralAttributes.CAPTION) );

                }else {
                    Element parent = (Element)attribute.getParentNode();
                    parent.removeChild(attribute);
                }
            }
        }
    }

    private boolean isConfigAttribute(String name) {
        return name.equals("config-ref");
    }


    private void parseProcessorChildElement(Element processor, Element attributeCategory)
            throws XPathExpressionException
    {
        logger.debug("- Parse Processor Child Element");

        NodeList processorChildElements = xPath.evaluateOnElementAndGetNodeList(processor, "./childElements/*");
        logger.debug("-- Found " + processorChildElements.getLength() + " processorChildElements");

        NodeList childs = xPath.evaluateOnElementAndGetNodeList(attributeCategory,"./*/*");
        for(int p=0; p<processorChildElements.getLength(); p++){
            Element processorChild  =  (Element) processorChildElements.item(p);

            for(int x=0; x<childs.getLength(); x++){
                String childname = ((Element) childs.item(x)).getAttribute(GeneralAttributes.NAME);

                logger.debug("--- Parsing " + childname + " child");
                if( childname.toLowerCase().contains(processorChild.getAttribute(GeneralAttributes.NAME))){
                    Element group = (Element) childs.item(x).getParentNode();
                    processorChild.setAttribute(GeneralAttributes.GROUP, group.getAttribute(GeneralAttributes.CAPTION) );
                }
            }

            addStudioPropertiesToChildElement(processorChild);
        }
    }


    public void addStudioPropertiesToChildElement(Element childElementFromSchema)
            throws XPathExpressionException
    {
        logger.debug("- Add Studio Properties to ChildElement");
        logger.debug("-- ChildElement " + childElementFromSchema.getAttribute(GeneralAttributes.NAME));

        Element nestedDefinition = nestedElements.get(childElementFromSchema.getAttribute(GeneralAttributes.NAME));

        // If child element has a definition as a nested element
        if(nestedDefinition != null){

            NodeList childAttribsFromSchema = xPath.evaluateOnElementAndGetNodeList(childElementFromSchema, "./" + OutputTagNames.ATTRIBUTES_NODE);
            // For each attribute node in the child elements node from schema
            for(int i=0; i < childAttribsFromSchema.getLength(); i++){

                Element attributesNode = (Element) childAttribsFromSchema.item(i);
                String groupController = attributesNode.getAttribute(GeneralAttributes.CONTROLLED);

                String expression = "./*[@name='"+ groupController +"']";
                Element controller = (Element) xPath.evaluateOnElementAndGetNode(nestedDefinition, expression);
                if(controller != null){
                    attributesNode.setAttribute(GeneralAttributes.CONTROLLER_TYPE, controller.getTagName());
                    attributesNode.setAttribute(GeneralAttributes.CAPTION, controller.getAttribute(GeneralAttributes.CAPTION));
                }

                parseChildElementAttributeFromAttributesNode(nestedDefinition, attributesNode);
            }

            NodeList childChildsFromSchema = xPath.evaluateOnElementAndGetNodeList(childElementFromSchema, "./" + OutputTagNames.CHILD_ELEMENTS_NODE);
            // For each attribute node in the child elements node from schema
            for(int i=0; i < childChildsFromSchema.getLength(); i++){

                Element childNode = (Element) childChildsFromSchema.item(i);
                addStudioPropertiesToChildElement(childNode);
            }

        } else {

            Element parent = (Element)childElementFromSchema.getParentNode();
            parent.removeChild(childElementFromSchema);
        }
    }

    private void parseChildElementAttributeFromAttributesNode(Element nestedDefinition, Element attributesNode)
            throws XPathExpressionException
    {
        logger.debug("- Parse Child Element Attribute From Attributes Node");

        String prefix = attributesNode.getAttribute(GeneralAttributes.CONTROLLED)
                        + attributesNode.getAttribute(GeneralAttributes.CAPTION)
                        + attributesNode.getAttribute(GeneralAttributes.CONTROLLER_TYPE);

        // For each attribute in the attributes node
        for(int p=0; p < attributesNode.getChildNodes().getLength(); p++){

            Element attribute = (Element) attributesNode.getChildNodes().item(p);
            String name = attribute.getAttribute(GeneralAttributes.NAME);

            if( !isConfigAttribute(name)){

                if(attributesIndex.containsKey(prefix+name))
                    copyNodeFromCachedNode(attribute, attributesIndex.get(prefix+name));

                else{
                    populateAttributesNodeFieldsFromNestedDefinition(nestedDefinition, attribute, name);

                    attributesIndex.put(prefix+name, attribute);
                }
            }
        }
    }

    private void populateAttributesNodeFieldsFromNestedDefinition(Element nestedDefinition, Element attribute, String name) throws XPathExpressionException {
        String nameExp = "./*[@name='"+ name +"']";
        Element field = (Element) xPath.evaluateOnElementAndGetNode(nestedDefinition,nameExp);

        if (field != null){
            copyFieldPropertiesToAttributeNode(attribute, field);

        }else{
            String baseName = attribute.getAttribute(GeneralAttributes.BASE);

            if(!baseName.equals("")){
                Element nestedBase = nestedElements.get(baseName);

                if(nestedBase!= null){
                    field = (Element) xPath.evaluateOnElementAndGetNode(nestedBase,nameExp);
                    if (field != null){
                        copyFieldPropertiesToAttributeNode(attribute, field);
                    }
                }
            }
        }
    }

    private void copyNodeFromCachedNode(Element emptyElement, Element cachedElement) {

        NamedNodeMap cachedAttributes = cachedElement.getAttributes();
        for (int i = 0; i < cachedAttributes.getLength(); ++i){
            Node attr = cachedAttributes.item(i);
            emptyElement.setAttribute(attr.getNodeName(), attr.getNodeValue());
        }
    }


    public void addStudioAttributePropertiesToGlobalConfig(Element configElementFromSchema)
            throws XPathExpressionException
    {
        NodeList globalElements = (editorsDoc.getDocumentElement().getElementsByTagName(EditorsTagNames.GLOBAL_CLOUD_CONNECTOR));
        for(int i=0; i < globalElements.getLength(); i++){

            Element cloudElem = (Element) globalElements.item(i);

            Element config = xPath.getFirstElementWithTag(configElementFromSchema, cloudElem.getAttribute(GeneralAttributes.LOCAL_ID));
            Element attributeCategory = xPath.getFirstElementWithTag(cloudElem, EditorsTagNames.ATTRIBUTE_CATEGORY);

            if (config != null && attributeCategory != null){

                setConnectorToolName(configElementFromSchema, cloudElem);

                NodeList configAttributes = xPath.evaluateOnElementAndGetNodeList(config, "./attributes/*");
                for(int p=0; p<configAttributes.getLength(); p++){

                    Element attribute = (Element) configAttributes.item(p);
                    String name = attribute.getAttribute(GeneralAttributes.NAME);

                    String expression = "./*/*[@name='"+ name +"']";

                    Element field = (Element) xPath.evaluateOnElementAndGetNode(attributeCategory, expression);
                    if (field != null){

                        copyFieldPropertiesToAttributeNode(attribute, field);

                        Element group = (Element)field.getParentNode();
                        attribute.setAttribute(GeneralAttributes.GROUP, group.getAttribute(GeneralAttributes.CAPTION) );

                    }else{

                        Element parent = (Element)attribute.getParentNode();
                        parent.removeChild(attribute);
                    }
                }
            }
        }
    }


    private void setConnectorToolName(Element configElementFromSchema, Element cloudElem) {

        String connectorName = cloudElem.getAttribute(GeneralAttributes.CAPTION);
        ((Element)configElementFromSchema.getOwnerDocument().getFirstChild()).setAttribute(GeneralAttributes.NAME,connectorName);
    }

}
