package com.mule.connectors.testdata.templates;

import com.mule.connectors.testdata.model.ConnectorProcessorInfo;
import com.mule.connectors.testdata.model.naming.PropertiesTagNames;
import com.mule.connectors.testdata.model.naming.GeneralAttributes;
import com.mule.connectors.testdata.model.naming.OutputTagNames;
import com.mule.connectors.testdata.utils.DocumentHandler;
import com.mule.connectors.testdata.utils.Utils;
import org.apache.log4j.Logger;
import org.w3c.dom.Element;
import org.w3c.dom.NodeList;

import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerException;
import javax.xml.xpath.XPath;
import javax.xml.xpath.XPathConstants;
import javax.xml.xpath.XPathExpressionException;
import javax.xml.xpath.XPathFactory;

import java.io.BufferedReader;
import java.io.FileReader;
import java.io.IOException;
import java.util.HashMap;
import java.util.Map;

public class PropertiesOverriderTemplate {

    private Logger logger = Logger.getLogger(PropertiesOverriderTemplate.class);

    private DocumentHandler docHandler;

    private Element root;
    private Element simpleConfig = null;
    private Element oauthConfig = null;

    private Element messageProcessors;
    private Element properties;
    private String outputFile;

    private XPath xPath = XPathFactory.newInstance().newXPath();

    public PropertiesOverriderTemplate(String connectorName, String originalOutputFile)
            throws ParserConfigurationException
    {

        logger.debug("- Properties Override Template initialize");

        this.outputFile = originalOutputFile.replace(".xml","-override.xml");
        this.docHandler = new DocumentHandler();
        this.root = docHandler.createRootElement(OutputTagNames.CONNECTOR);
        root.setAttribute(GeneralAttributes.NAME, connectorName);

    }


    public void setPropertiesElement(Element connectorProperties){

        logger.info("-- Setting Connector Properties Element");

        // Override with custom properties
        if(connectorProperties != null){
            this.properties = (Element) docHandler.importNode(connectorProperties, true);

        }else{

            createConnectorDefaultProperties();
        }
    }

    private void createConnectorDefaultProperties() {

        logger.debug("- Create Connector Default Properties");

        this.properties = docHandler.createElement(OutputTagNames.PROPERTIES);

        Map<String, String> defaultProps = new HashMap<String, String>();
        defaultProps.put(PropertiesTagNames.AUTHENTICATION_OAUTH, "false");
        defaultProps.put(PropertiesTagNames.CONNECTIVITY_SUPPORT, "true");
        defaultProps.put(PropertiesTagNames.DATASENSE_ENABLED, "true");
        defaultProps.put(PropertiesTagNames.METADATA_TYPE, "static");
        defaultProps.put(PropertiesTagNames.IS_TRANSFORMER, "false");

        for(String key: defaultProps.keySet()){
            Element node = docHandler.createElement(key);
            node.setTextContent(defaultProps.get(key));

            this.properties.appendChild(node);
        }
    }

    public void setMessageProcessorsElement(Element processors,
                                            Map<String,ConnectorProcessorInfo> processorsProperties)
            throws XPathExpressionException
    {
        logger.info("-- Merging Processors Elements");
        logger.debug("- Set Message Processors Element");

        this.messageProcessors = docHandler.createElement( OutputTagNames.MESSAGE_PROCESSORS_NODE);

        NodeList childs = processors.getChildNodes();
        logger.debug("-- Found " + childs.getLength() + " procesors");

        // Parse all the processors present in the connector
        for(int i=0; i < childs.getLength(); i++){

            Element processorFromSchema = (Element) childs.item(i);
            String processorName = processorFromSchema.getTagName();

            logger.debug("--- Creating " + processorName + " processor element");

            Element processor = docHandler.createElement(processorName);
            processor.setAttribute(GeneralAttributes.NAME, Utils.toHumanReadable(processorFromSchema.getTagName()));

            setProcessorProperties(processor, processorsProperties.get(processorName));

            Element attributes = importRequiredAttributesFromSchemaOutput(processorFromSchema);
            processor.appendChild(attributes);

            Element childElem = importChildElementsFromSchemaOutput(processorFromSchema, OutputTagNames.REQUIRED);
            processor.appendChild(childElem);

            this.messageProcessors.appendChild(processor);
        }
    }

    private Element importRequiredAttributesFromSchemaOutput(Element processorFromSchema)
            throws XPathExpressionException
    {
        logger.debug("- Import Required Attributes From Schema Output");

        // Set required attributes
        Element attributes = docHandler.createElement(OutputTagNames.ATTRIBUTES_NODE);
        NodeList required = (NodeList) xPath.compile("./attributes/required").evaluate(processorFromSchema, XPathConstants.NODESET);
        for (int node=0; node<required.getLength(); node++){
            Element attr = (Element) docHandler.importNode(required.item(node), false);
            // Adding " " as content results in open-close tags instead of simple closed tag
            attr.setTextContent(" ");
            attributes.appendChild(attr);
        }
        return attributes;
    }


    private void setProcessorProperties(Element processor, ConnectorProcessorInfo processorProperties){

        logger.debug("- Set Properties for Processor :: " + processor.getTagName());

        if(processorProperties != null){

            Element dmap = docHandler.createElement(OutputTagNames.DATAMAPPER_NODE);
            dmap.setAttribute(GeneralAttributes.DM_OUTPUT, processorProperties.getOuputDataMapperType());
            dmap.setAttribute(GeneralAttributes.DM_INPUT, processorProperties.getInputDataMapperType());
            processor.appendChild(dmap);

            Element pag = docHandler.createElement(PropertiesTagNames.AUTO_PAGING);
            pag.setTextContent(processorProperties.hasAutoPaging().toString());
            processor.appendChild(pag);

            Element query = docHandler.createElement(PropertiesTagNames.QUERY_SUPPORT);
            query.setTextContent(processorProperties.hasQuerySupport().toString());
            processor.appendChild(query);

        }else{
            logger.debug("-- Using Default Properties");

            createProcessorDefaultProperties(processor);
        }

    }


    private void createProcessorDefaultProperties(Element processor) {

        logger.debug("- Create Default Properties");

        Element dmap = docHandler.createElement(OutputTagNames.DATAMAPPER_NODE);
        dmap.setAttribute(GeneralAttributes.DM_OUTPUT, "");
        dmap.setAttribute(GeneralAttributes.DM_INPUT, "");
        processor.appendChild(dmap);

        Element pag = docHandler.createElement(PropertiesTagNames.AUTO_PAGING);
        pag.setTextContent("false");
        processor.appendChild(pag);

        Element query = docHandler.createElement(PropertiesTagNames.QUERY_SUPPORT);
        query.setTextContent("false");
        processor.appendChild(query);

    }


    private Element importChildElementsFromSchemaOutput(Element source, String type)
            throws XPathExpressionException
    {
        logger.debug("- Import ChildElements From SchemaOutput");

        Element childElem = docHandler.createElement(OutputTagNames.CHILD_ELEMENTS_NODE);

        NodeList nodes = (NodeList) xPath.compile("./childElements/"+type).evaluate(source, XPathConstants.NODESET);

        logger.debug("-- Found " + nodes.getLength() + " childs");

        for (int nodeIndx=0; nodeIndx < nodes.getLength(); nodeIndx++){
            Element child = importChildElement((Element) nodes.item(nodeIndx));
            childElem.appendChild(child);
        }

        return childElem;
    }

    private Element importChildElement(Element source) throws XPathExpressionException {

        logger.debug("- Import ChildElement From Source");

        Element parentChild = docHandler.createElement(source.getTagName());

        parentChild.setAttribute(GeneralAttributes.NAME, source.getAttribute(GeneralAttributes.NAME));
        parentChild.setAttribute(GeneralAttributes.GROUP, source.getAttribute(GeneralAttributes.GROUP));


        NodeList attributesSources = (NodeList) xPath.compile("./attributes").evaluate(source, XPathConstants.NODESET);

        for(int i= 0; i<attributesSources.getLength(); i++){
            Element attributesSource = (Element) attributesSources.item(i);

            Element attributes = docHandler.createElement(OutputTagNames.ATTRIBUTES_NODE);
            attributes.setAttribute(GeneralAttributes.CAPTION, attributesSource.getAttribute(GeneralAttributes.CAPTION));
            attributes.setAttribute(GeneralAttributes.CONTROLLED, attributesSource.getAttribute(GeneralAttributes.CONTROLLED));
            attributes.setAttribute(GeneralAttributes.CONTROLLER_TYPE, attributesSource.getAttribute(GeneralAttributes.CONTROLLER_TYPE));

            NodeList required = (NodeList) xPath.compile("./required").evaluate(attributesSource, XPathConstants.NODESET);
            for (int node=0; node<required.getLength(); node++){
                Element attr = (Element) docHandler.importNode(required.item(node), false);
                attr.setTextContent(" ");
                attributes.appendChild(attr);
            }
            parentChild.appendChild(attributes);

        }


        Element childElem = importChildElementsFromSchemaOutput(source, OutputTagNames.REQUIRED);
        parentChild.appendChild(childElem);

        return parentChild;
    }


    private Element cloneNode(Element origin, String tagName){

        Element node = docHandler.createElement(tagName);
        node.setAttribute(GeneralAttributes.CAPTION, origin.getAttribute(GeneralAttributes.CAPTION));
        node.setAttribute(GeneralAttributes.NAME, origin.getAttribute(GeneralAttributes.NAME));
        node.setAttribute(GeneralAttributes.TYPE, origin.getAttribute(GeneralAttributes.TYPE));
        node.setAttribute(GeneralAttributes.JAVATYPE, origin.getAttribute(GeneralAttributes.JAVATYPE));
        node.setAttribute(GeneralAttributes.GROUP, origin.getAttribute(GeneralAttributes.GROUP));
        node.setTextContent(" ");

        return node;
    }


    public void exportToFile() throws TransformerException {

        logger.debug("- Export to File");

        buildOutputDocument();
        docHandler.exportToFile(outputFile);
    }


    private void buildOutputDocument(){

        logger.debug("- Build Output Document");

        root.appendChild(properties);

        if(simpleConfig != null)
            root.appendChild(simpleConfig);
        if(oauthConfig != null)
            root.appendChild(oauthConfig);

        root.appendChild(messageProcessors);
    }




    public void setConfigElement(Element globalConfigElement, String credentialsFile)
            throws IOException, XPathExpressionException
    {
        logger.debug("- Set Config Element");
        logger.info("-- Setting Connector Config Element");
        if(globalConfigElement.getElementsByTagName("config").getLength() > 0){
            logger.debug("-- Creating Simple Config Override");

            this.simpleConfig = docHandler.createElement(OutputTagNames.CONFIG_NODE);
            Element userPswdConfig = (Element) globalConfigElement.getElementsByTagName("config").item(0);

            setConfigNameAttribute(userPswdConfig, this.simpleConfig);

            if( !credentialsFile.equals(" ") ){
                parseCredentialsFromFile(userPswdConfig, this.simpleConfig, credentialsFile);
            }else{
                createDefaultConfig(userPswdConfig, this.simpleConfig);
            }

        }

        if(globalConfigElement.getElementsByTagName("config-with-oauth").getLength() > 0){
            logger.debug("-- Creating OAuth Config Override");

            this.oauthConfig = docHandler.createElement(OutputTagNames.OAUTH_NODE);
            Element oauth = (Element) globalConfigElement.getElementsByTagName("config-with-oauth").item(0);

            setConfigNameAttribute(oauth, this.oauthConfig);

            if( !credentialsFile.equals(" ") ){
                parseCredentialsFromFile(oauth, this.oauthConfig, credentialsFile);
            }else{
                createDefaultConfig(oauth, this.oauthConfig);
            }
        }
    }

    private void setConfigNameAttribute(Element source, Element destination)
            throws XPathExpressionException
    {
        String nameExpression = "./attributes/*[@name='name']";
        Element origin = (Element) xPath.compile(nameExpression).evaluate(source, XPathConstants.NODE);
        if(origin != null){
            Element node = cloneNode(origin, OutputTagNames.REQUIRED);
            node.setTextContent(" ");
            destination.appendChild(node);
        }

    }

    private void createDefaultConfig(Element source, Element destination) {

        logger.debug("- Create Default Config");

        // Clone required fields
        NodeList required = source.getElementsByTagName(OutputTagNames.REQUIRED);
        for(int i=0; i<required.getLength(); i++){
            Element node = (Element) docHandler.importNode(required.item(i), false);
            node.setAttribute(GeneralAttributes.PREFIX, "");
            node.setTextContent(" ");

            destination.appendChild(node);
        }
    }


    private void parseCredentialsFromFile(Element source, Element destination, String file)
            throws IOException, XPathExpressionException
    {
        logger.info("-- Parsing Credentials From File");
        logger.debug("- Parse Credentials From File");

        BufferedReader br = null;
        try {

            br = new BufferedReader(new FileReader(file));
            String currentLine;

            while ((currentLine = br.readLine()) != null) {

                createRequiredNodeFromCredential(source, destination, currentLine);
            }

        } catch (IOException e) {
            throw e;
        } catch (XPathExpressionException e) {
            throw e;
        } finally {
            if (br != null) br.close();
        }

    }

    private void createRequiredNodeFromCredential(Element source, Element destination, String currentLine)
            throws XPathExpressionException
    {
        logger.debug("- Create RequiredNode From Credentials");

        int indexOfDot = currentLine.indexOf(".");
        int indexOfEqual = currentLine.indexOf("=");

        String prefix = (( indexOfDot != -1) ? currentLine.substring(0, indexOfDot) : "") ;

        String fieldName = currentLine.substring(indexOfDot + 1, indexOfEqual);
        String value = currentLine.substring(indexOfEqual + 1);

        String attrWithNameExpression = "./attributes/*[@name='"+ fieldName +"']";
        Element origin = (Element) xPath.compile(attrWithNameExpression).evaluate(source, XPathConstants.NODE);

        if(origin != null){
            logger.debug("-- Read attribute :: " + fieldName);

            Element node = cloneNode(origin, OutputTagNames.REQUIRED);
            node.setAttribute(GeneralAttributes.PREFIX, prefix);
            node.setTextContent(value);

            destination.appendChild(node);
        }
    }

}
