/**
 * This module contains DataWeave functions for filtering the api graph in JSON-LD format.
 */
%dw 2.0
import every from dw::core::Arrays
import toBoolean from dw::util::Coercions

// -------- Util functions --------
fun getObjectById(objectsList,id) = objectsList filter ((item, index) -> item."@id" == id)
fun matchingObjectsById(objectsList, listOfIds) = listOfIds flatMap getObjectById(objectsList,$)
fun flattenIds(idsArray) = idsArray reduce ((idObject, accumulator = []) -> accumulator ++ idObject.*"@id")
fun isOfTypes(object, matchingTypes) = matchingTypes every (object."@type" contains $)
fun filterTypes(objects,matchingTypes) = objects filter ((obj, index) -> isOfTypes(obj,matchingTypes))
fun excludeTypes(objects,matchingTypes) = objects filter ((obj, index) -> not isOfTypes(obj,matchingTypes))

// -------- Util functions for getting collections from the api graph --------
fun getEndpoints(api) = (api."@graph" filterTypes ["apiContract:EndPoint"])
fun getQueries(api) = (api."@graph" filterTypes ["apiContract:EndPoint"]) filter ((endpoint) -> (endpoint."apiContract:path" startsWith "/query"))
fun getMutations(api) = (api."@graph" filterTypes ["apiContract:EndPoint"]) filter ((endpoint) -> (endpoint."apiContract:path" startsWith "/mutation"))
fun getOperations(api) = api."@graph" filterTypes ["apiContract:Operation"]
fun getResponses(api) = api."@graph" filterTypes ["apiContract:Response","apiContract:Message"]
fun getRequests(api) = api."@graph" filterTypes ["apiContract:Request","apiContract:Message"]
fun getMessagePayloads(api) = api."@graph" filterTypes ["apiContract:Payload"]
fun getParameters(api) = api."@graph" filterTypes ["apiContract:Parameter"]
fun getShapes(api) = api."@graph" filterTypes ["raml-shapes:Shape"]
fun getPayloads(api) = api."@graph" filterTypes ["raml-shapes:Payload"]
fun getNodeShapeTypes(api) = getShapes(api) filterTypes ["shacl:NodeShape"] filter ((shape) -> !toBoolean(shape."raml-shapes:inputOnly" default false) and !toBoolean(shape."raml-shapes:isAbstract" default false))
fun getPropertyShapes(api) = getShapes(api) filterTypes ["shacl:PropertyShape"]
fun getNonNilShapes(api) = getShapes(api) filter ((shape) -> !(shape."@type" contains "raml-shapes:NilShape"))
fun getOperationsNames(api) = getOperations(api).*"apiContract:operationId"

// -------- Util functions to seek objects in the api graph --------
// Get endpoint's operations distinguished by method
fun getEndpointOperations(api,endpoint) = matchingObjectsById(api."@graph", flattenIds(endpoint."apiContract:supportedOperation")) filter ((operation, index) -> !isEmpty(operation."apiContract:method")) distinctBy ((operation, index) -> operation."apiContract:method")
// Get operation's request
fun getOperationRequest(api,operation) = matchingObjectsById(getRequests(api), flattenIds(operation."apiContract:expects"))
// Get operation's response
fun getOperationResponse(api,operation) = matchingObjectsById(getResponses(api),flattenIds(operation."apiContract:returns"))
// Get message's payloads
fun getMessagePayloads(api,response) = matchingObjectsById(getMessagePayloads(api), flattenIds(response."apiContract:payload"))
// Get schemas from payload filtering out those that are of type NilShape
fun getNonNilSchemasFromPayload(api,payload) = matchingObjectsById(getShapes(api), flattenIds(payload.*"raml-shapes:schema")) flatMap ((schema) -> matchingObjectsById(getNonNilShapes(api),flattenIds(schema."raml-shapes:anyOf")))
// Get properties from schema
fun getPropertiesFromSchema(api,schema) = matchingObjectsById(getPropertyShapes(api), flattenIds(schema."shacl:property")) default []
// Get range from property
fun getNotScalarPropertyTypes(api,property) = matchingObjectsById(api."@graph", flattenIds(property.*"raml-shapes:range")) excludeTypes ["raml-shapes:ScalarShape"]
// Returns the NodeShape wrapped by an ArrayShape or UnionShape
fun getNodeShapeFromCompoundShape(api,schema) = schema match {
    case s if (isOfTypes(schema,["raml-shapes:ArrayShape"])) -> (
        matchingObjectsById(api."@graph",flattenIds(schema.*"raml-shapes:items"))
        flatMap ((itemSchema, index) -> getNodeShapeFromCompoundShape(api,itemSchema))
    )
    case s if (isOfTypes(schema,["raml-shapes:UnionShape"])) -> (
        matchingObjectsById(api."@graph",flattenIds(schema.*"raml-shapes:anyOf"))
        flatMap ((unionSchema, index) -> getNodeShapeFromCompoundShape(api,unionSchema))
    )
    case s if (isOfTypes(schema,["shacl:NodeShape"])) -> schema
    else -> {}
}
// Get request's parameters
fun getRequestParameters(api,request) = matchingObjectsById(getParameters(api),flattenIds(request."apiContract:parameter"))
// Get range from property
fun getPropertyTypes(api,property) = matchingObjectsById(api."@graph", flattenIds(property.*"raml-shapes:range"))