/**
* This module contains helper functions for working with strings.
*
* To use this module, you must import it to your DataWeave code, for example,
* by adding the line `import * from dw::core::Strings` to the header of your
* DataWeave script.
*/
%dw 2.0

/**
* Returns the Unicode for the first character in an input string.
*
*
* For an empty string, the function fails and returns `Unexpected empty string`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* |===
*
* === Example
*
* This example returns Unicode for the "M" in "Mule".
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "charCode" : charCode("Mule")
* }
* ----
*
* ==== Output
*
* [source,XML,linenums]
* ----
* { "charCode" : 77 }
* ----
*/
fun charCode(text: String): Number = native("system::CharCodeFunctionValue")

/**
* Helper function that enables `charCode` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun charCode(text: Null): Null = null

/**
* Returns the Unicode for a character at the specified index.
*
*
* This function fails if the index is invalid.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `content` | The input string.
* | `position` | The index (a `Number` type) of a character in the string (as a string array). Note that the index of the first character is `0`.
* |===
*
* === Example
*
* This example returns Unicode for the "u" at index `1` in "MuleSoft".
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "charCodeAt" : charCodeAt("MuleSoft", 1)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { "charCodeAt": 117 }
* ----
*/
fun charCodeAt(content: String, position: Number): Number =
  charCode(content[position]!)

/**
* Helper function that enables `charCodeAt` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun charCodeAt(content: Null, position: Any): Null = null

/**
* Returns a character that matches the specified Unicode.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `charCode` | The input Unicode (a `Number`).
* |===
*
* === Example
*
* This example inputs the Unicode number `117` to return the character "u".
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "fromCharCode" : fromCharCode(117)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { "fromCharCode": "u" }
* ----
*/
fun fromCharCode(charCode: Number): String = native("system::FromCharCodeFunctionValue")

/**
* Helper function that enables `fromCharCode` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun fromCharCode(charCode: Null): Null = null

/**
* Pluralizes a singular string.
*
*
* If the input is already plural (for example, "boxes"), the output will match
* the input.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to pluralize.
* |===
*
* === Example
*
* This example pluralizes the input string "box" to return "boxes".
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
*  { "pluralize" : pluralize("box") }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { "pluralize" : "boxes" }
* ----
**/
fun pluralize(text: String): String = native("system::StringPluralizeOperator")

/**
* Helper function that enables `pluralize` to work with a `null` value.
*/
fun pluralize(text: Null): Null = null

/**
* Converts a plural string to its singular form.
*
*
* If the input is already singular (for example, "box"), the output will match
* the input.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to convert to singular form.
* |===
*
* === Example
*
* This example converts the input string "boxes" to return "box".
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* { "singularize" : singularize("boxes") }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* { "singularize" : "box" }
* ----
**/
fun singularize(text: String): String = native("system::StringSingularizeFunctionValue")

/**
* Helper function that enables `singularize` to work with a `null` value.
*/
fun singularize(text: Null): Null = null

/**
* Returns a string in camel case based on underscores in the string.
*
*
* All underscores are deleted, including any underscores at the beginning of the string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to convert to camel case.
* |===
*
* === Example
*
* This example converts a string that contains underscores to camel case.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : camelize("customer_first_name"),
*   "b" : camelize("_name_starts_with_underscore")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*    "a": "customerFirstName",
*    "b": "nameStartsWithUnderscore"
*  }
* ----
**/
fun camelize(text: String): String = native("system::StringCamelizeFunctionValue")

/**
* Helper function that enables `camelize` to work with a `null` value.
*/
fun camelize(text: Null): Null = null

/**
* Capitalizes the first letter of each word in a string.
*
*
* It also removes underscores between words and puts a space before each
* capitalized word.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string to capitalize.
* |===
*
* === Example
*
* This example capitalizes a set of strings.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : capitalize("customer"),
*   "b" : capitalize("customer_first_name"),
*   "c" : capitalize("customer NAME"),
*   "d" : capitalize("customerName")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": "Customer",
*   "b": "Customer First Name",
*   "c": "Customer Name",
*   "d": "Customer Name"
* }
* ----
**/
fun capitalize(text: String): String = native("system::StringCapitalizeFunctionValue")

/**
* Helper function that enables `capitalize` to work with a `null` value.
*/
fun capitalize(text: Null): Null = null

/**
* Returns a number as an ordinal, such as `1st` or `2nd`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `num` | An input number to return as an ordinal.
* |===
*
* === Example
*
* This example returns a variety of input numbers as ordinals.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : ordinalize(1),
*   "b": ordinalize(2),
*   "c": ordinalize(5),
*   "d": ordinalize(103)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*    "a": "1st",
*    "b": "2nd",
*    "c": "5th",
*    "d": "103rd"
* }
* ----
**/
fun ordinalize(num: Number): String = native("system::NumberOrdinalizeFunctionValue")

/**
* Helper function that enables `ordinalize` to work with a `null` value.
*/
fun ordinalize(num: Null): Null = null

/**
* Replaces hyphens, spaces, and camel-casing in a string with underscores.
*
*
* If no hyphens, spaces, and camel-casing are present, the output will match
* the input.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* |===
*
* === Example
*
* This example replaces the hyphens and spaces in the input. Notice that
* the input "customer" is not modified in the output.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*    "a" : underscore("customer"),
*    "b" : underscore("customer-first-name"),
*    "c" : underscore("customer NAME"),
*    "d" : underscore("customerName")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*    "a": "customer",
*    "b": "customer_first_name",
*    "c": "customer_name",
*    "d": "customer_name"
* }
* ----
**/
fun underscore(text: String): String = native("system::StringUnderscoreFunctionValue")

/**
* Helper function that enables `underscore` to work with a `null` value.
*/
fun underscore(text: Null): Null = null

/**
* Replaces spaces, underscores, and camel-casing in a string with dashes
* (hyphens).
*
*
* If no spaces, underscores, and camel-casing are present, the output will
* match the input.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* |===
*
* === Example
*
* This example replaces the spaces, underscores, and camel-casing in the input.
* Notice that the input "customer" is not modified in the output.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a" : dasherize("customer"),
*   "b" : dasherize("customer_first_name"),
*   "c" : dasherize("customer NAME"),
*   "d" : dasherize("customerName")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": "customer",
*   "b": "customer-first-name",
*   "c": "customer-name",
*   "d": "customer-name"
* }
* ----
**/
fun dasherize(text: String): String = native("system::StringDasherizeFunctionValue")

/**
* Helper function that enables `dasherize` to work with a `null` value.
*/
fun dasherize(text: Null): Null = null

/**
* The specified `text` is _left_-padded to the `size` using the `padText`.
* By default `padText` is `" "`.
*
*
* Returns left-padded `String` or original `String` if no padding is necessary.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `size` | The size to pad to.
* | `padText` | The text to pad with. It defaults to one space if not specified.
* |===
*
* === Example
*
* This example shows how `leftPad` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*    "a": leftPad(null, 3),
*    "b": leftPad("", 3),
*    "c": leftPad("bat", 5),
*    "d": leftPad("bat", 3),
*    "e": leftPad("bat", -1)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": null,
*   "b": "   ",
*   "c": "  bat",
*   "d": "bat",
*   "e": "bat"
* }
* ----
*/
@Since(version = "2.2.0")
fun leftPad(text: String, size: Number, padText: String = " "): String = do {
    var actualSize = sizeOf(text)
    ---
    if (actualSize >= size)
        text
    else
        padText repeat (size - actualSize) ++ text
}

/**
* Helper function that enables `leftPad` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun leftPad(text: Null, size: Any, padText: Any = " "): Null = null

/**
* The specified `text` is _right_-padded to the `size` using the `padText`.
* By default `padText` is `" "`.
*
*
* Returns right padded `String` or original `String` if no padding is necessary.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `size` | The size to pad to.
* | `padText` | The text to pad with. It defaults to one space if not specified.
* |===
*
* === Example
*
* This example shows how `rightPad` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": rightPad(null, 3),
*   "b": rightPad("", 3),
*   "c": rightPad("bat", 5),
*   "d": rightPad("bat", 3),
*   "e": rightPad("bat", -1)
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": null,
*   "b": "   ",
*   "c": "bat  ",
*   "d": "bat",
*   "e": "bat"
* }
* ----
*/
@Since(version = "2.2.0")
fun rightPad(text: String, size: Number, padChar: String = " "): String =  do {
    var actualSize = sizeOf(text)
    ---
    if (actualSize >= size)
       text
    else
       text ++ (padChar repeat (size - actualSize))
}

/**
* Helper function that enables `rightPad` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun rightPad(text: Null, size: Any, padText: Any = " "): Null = null

/**
* Wraps the specified `text` with the given `wrapper`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `wrapper` | The content used to wrap.
* |===
*
* === Example
*
* This example shows how `wrapWith` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": wrapWith(null, "'"),
*   "b": wrapWith("", "'"),
*   "c": wrapWith("ab", "x"),
*   "d": wrapWith("'ab'", "'"),
*   "e": wrapWith("ab", "'")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": null,
*   "b": "''",
*   "c": "xabx",
*   "d": "''ab''",
*   "e": "'ab'"
* }
* ----
*/
@Since(version = "2.2.0")
fun wrapWith(text: String, wrapper: String): String =
    "$(wrapper)$(text)$(wrapper)"

/**
* Helper function that enables `wrapWith` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun wrapWith(text: Null, wrapper: Any): Null = null

/**
* Wraps `text` with `wrapper` if that `wrapper` is missing from the start or
* end of the given string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `wrapper` | The content used to wrap.
* |===
*
* === Example
*
* This example shows how `wrapIfMissing` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
*  {
*    "a": wrapIfMissing(null, "'"),
*    "b": wrapIfMissing("", "'"),
*    "c": wrapIfMissing("ab", "x"),
*    "d": wrapIfMissing("'ab'", "'"),
*    "e": wrapIfMissing("/", '/'),
*    "f": wrapIfMissing("a/b/c", '/'),
*    "g": wrapIfMissing("/a/b/c", '/'),
*    "h": wrapIfMissing("a/b/c/", '/')
*  }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*    "a": null,
*    "b": "'",
*    "c": "xabx",
*    "d": "'ab'",
*    "e": "/",
*    "f": "/a/b/c/",
*    "g": "/a/b/c/",
*    "h": "/a/b/c/"
*  }
* ----
*/
@Since(version = "2.2.0")
fun wrapIfMissing(text: String, wrapper: String): String = do {
    var withHead = if(!startsWith(text, wrapper)) "$(wrapper)$(text)" else text
    ---
    if (!endsWith(withHead, wrapper))
        "$(withHead)$(wrapper)"
    else
        withHead
}

/**
* Helper function that enables `wrapIfMissing` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun wrapIfMissing(text: Null, wrapper: String): Null = text

/**
* Gets the substring before the first occurrence of a separator. The separator
* is not returned.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `separator` | String to search for.
* |===
*
* === Example
*
* This example shows how `substringBefore` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": substringBefore(null, "'"),
*   "b": substringBefore("", "-"),
*   "c": substringBefore("abc", "b"),
*   "d": substringBefore("abc", "c"),
*   "e": substringBefore("abc", "d"),
*   "f": substringBefore("abc", "")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": null,
*   "b": "",
*   "c": "a",
*   "d": "ab",
*   "e": "",
*   "f": ""
* }
* ----
*/
@Since(version = "2.2.0")
fun substringBefore(text: String, separator: String): String = do {
    var index = (text find separator)[0]
    ---
    index match {
        case is Null         -> ""
        case 0               -> ""
        case index is Number ->
            text[0 to index - 1] default ""
    }
}

/**
* Helper function that enables `substringBefore` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun substringBefore(text: Null, separator: String): Null = null

/**
* Gets the substring before the last occurrence of a separator. The separator
* is not returned.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `separator` | String to search for.
* |===
*
* === Example
*
* This example shows how `substringBeforeLast` behaves with different inputs
* and sizes.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": substringBeforeLast(null, "'"),
*   "b": substringBeforeLast("", "-"),
*   "c": substringBeforeLast("abc", "b"),
*   "d": substringBeforeLast("abcba", "b"),
*   "e": substringBeforeLast("abc", "d"),
*   "f": substringBeforeLast("abc", "")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*   "a": null,
*   "b": "",
*   "c": "a",
*   "d": "abc",
*   "e": "",
*   "f": "ab"
* }
* ----
*/
@Since(version = "2.2.0")
fun substringBeforeLast(text: String, separator: String): String = do {
    var index = (text find separator)[-1]
    ---
    index match {
        case is Null         -> ""
        case 0               -> ""
        case index is Number ->
            text[0 to index - 1] default ""
    }
}

/**
* Helper function that enables `substringBeforeLast` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun substringBeforeLast(text: Null, separator: String): Null = null


/**
* Gets the substring after the first occurrence of a separator. The separator
* is not returned.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `separator` | String to search for.
* |===
*
* === Example
*
* This example shows how `substringAfter` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": substringAfter(null, "'"),
*   "b": substringAfter("", "-"),
*   "c": substringAfter("abc", "b"),
*   "d": substringAfter("abcba", "b"),
*   "e": substringAfter("abc", "d"),
*   "f": substringAfter("abc", "")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*
*   "a": null,
*   "b": "",
*   "c": "c",
*   "d": "cba",
*   "e": "",
*   "f": "bc"
* }
* ----
*/
@Since(version = "2.2.0")
fun substringAfter(text: String, separator: String): String = do {
  var index = (text find separator)[0]
  ---
  index match {
      case is Null -> ""
      case index is Number ->
          text[index + sizeOf(separator) to -1] default ""
  }
}

/**
* Helper function that enables `substringAfter` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun substringAfter(text: Null, separator: String): Null = null

/**
* Gets the substring after the last occurrence of a separator. The separator
* is not returned.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The input string.
* | `separator` | String to search for.
* |===
*
* === Example
*
* This example shows how `substringAfterLast` behaves with different inputs and sizes.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* {
*   "a": substringAfterLast(null, "'"),
*   "b": substringAfterLast("", "-"),
*   "c": substringAfterLast("abc", "b"),
*   "d": substringAfterLast("abcba", "b"),
*   "e": substringAfterLast("abc", "d"),
*   "f": substringAfterLast("abc", "")
* }
* ----
*
* ==== Output
*
* [source,JSON,linenums]
* ----
* {
*  "a": null,
*  "b": "",
*  "c": "c",
*  "d": "a",
*  "e": "",
*  "f": null
* }
* ----
*/
@Since(version = "2.2.0")
fun substringAfterLast(text: String, separator: String): String = do {
    var index = (text find separator)[-1]
    ---
    if(isEmpty(separator))
        ""
    else
        index match {
            case is Null -> ""
            case index is Number ->
                text[index + sizeOf(separator) to -1] default ""
        }
}

/**
* Helper function that enables `substringAfterLast` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun substringAfterLast(text: Null, separator: String): Null = null


 /**
 * Repeats a `text` the number of specified `times`.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * | `times` | Number of times to repeat char. Negative is treated as zero.
 * |===
 *
 * === Example
 *
 * This example shows how `repeat` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import * from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": repeat("e", 0),
 *   "b": repeat("e", 3),
 *   "c": repeat("e", -2)
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": "",
 *   "b": "eee",
 *   "c": ""
 * }
 * ----
 */
@Since(version = "2.2.0")
fun repeat(text: String, times: Number): String =
    if(times <= 0) "" else (1 to times) map text joinBy ""

/**
* Helper function that enables `repeat` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun repeat(text: Null, times: Any): Null = null

/**
 * Unwraps a given `text` from a `wrapper` text.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * | `wrapper` | The text used to unwrap.
 * |===
 *
 * === Example
 *
 * This example shows how `unwrap` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import unwrap from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": unwrap(null, ""),
 *   "b": unwrap(null, '\0'),
 *   "c": unwrap("'abc'", "'"),
 *   "d": unwrap("AABabcBAA", 'A'),
 *   "e": unwrap("A", '#'),
 *   "f": unwrap("#A", '#'),
 *   "g": unwrap("A#", '#')
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": null,
 *   "b": null,
 *   "c": "abc",
 *   "d": "ABabcBA",
 *   "e": "A",
 *   "f": "A#",
 *   "g": "#A"
 * }
 * ----
 */
@Since(version = "2.2.0")
fun unwrap(text: String, wrapper: String): String = do {
    if(!startsWith(text, wrapper) and !endsWith(text, wrapper))
        text
    else
        text[1 to -2] default text
}

/**
* Helper function that enables `unwrap` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun unwrap(text: Null, wrapper: String): Null = null

/**
 * Prepends the `prefix` to the beginning of the string if the `text` does not
 * already start with that prefix.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * | `prefix` | The text to use as prefix.
 * |===
 *
 * === Example
 *
 * This example shows how `prependIfMissing` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import prependIfMissing from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": prependIfMissing(null, ""),
 *   "b": prependIfMissing("abc", ""),
 *   "c": prependIfMissing("", "xyz"),
 *   "d": prependIfMissing("abc", "xyz"),
 *   "e": prependIfMissing("xyzabc", "xyz")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": null,
 *   "b": "abc",
 *   "c": "xyz",
 *   "d": "xyzabc",
 *   "e": "xyzabc"
 * }
 * ----
 */
@Since(version = "2.2.0")
fun prependIfMissing(text: String, prefix: String): String = do {
    if(!startsWith(text, prefix))
        "$(prefix)$(text)"
    else
        "$(text)"
}

/**
* Helper function that enables `prependIfMissing` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun prependIfMissing(text: Null, prefix: String): Null = null

/**
 * Appends the `suffix` to the end of the `text` if the `text` does not already
 * ends with the `suffix`.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * | `suffix` | The text used as the suffix.
 * |===
 *
 * === Example
 *
 * This example shows how `appendIfMissing` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import appendIfMissing from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": appendIfMissing(null, ""),
 *   "b": appendIfMissing("abc", ""),
 *   "c": appendIfMissing("", "xyz") ,
 *   "d": appendIfMissing("abc", "xyz") ,
 *   "e": appendIfMissing("abcxyz", "xyz")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": null,
 *   "b": "abc",
 *   "c": "xyz",
 *   "d": "abcxyz",
 *   "e": "abcxyz"
 * }
 * ----
 */
@Since(version = "2.2.0")
fun appendIfMissing(text:String, suffix: String): String = do {
    if(!endsWith(text, suffix))
      "$(text)$(suffix)"
    else
      text
}

/**
* Helper function that enables `appendIfMissing` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun appendIfMissing(text: Null, suffix: String): Null = null

/**
 * Checks if the `text` contains only Unicode digits.
 *
 *
 * A decimal point is not a Unicode digit and returns false.
 * Note that the method does not allow for a leading sign, either positive or
 * negative.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isNumeric` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import isNumeric from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isNumeric(null),
 *   "b": isNumeric(""),
 *   "c": isNumeric("  "),
 *   "d": isNumeric("123"),
 *   "e": isNumeric("१२३"),
 *   "f": isNumeric("12 3"),
 *   "g": isNumeric("ab2c"),
 *   "h": isNumeric("12-3"),
 *   "i": isNumeric("12.3"),
 *   "j": isNumeric("-123"),
 *   "k": isNumeric("+123")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": true,
 *   "f": false,
 *   "g": false,
 *   "h": false,
 *   "i": false,
 *   "j": false,
 *   "k": false
 * }
 * ----
 */
@Since(version = "2.2.0")
fun isNumeric(text: String): Boolean = native("system::IsNumericStringFunctionValue")

/**
* Helper function that enables `isNumeric` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun isNumeric(text: Null): Boolean = false

/**
 * Checks if the `text` contains only whitespace.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isWhitespace` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import isWhitespace from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isWhitespace(null),
 *   "b": isWhitespace(""),
 *   "c": isWhitespace("  "),
 *   "d": isWhitespace("abc"),
 *   "e": isWhitespace("ab2c"),
 *   "f": isWhitespace("ab-c")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": true,
 *   "c": true,
 *   "d": false,
 *   "e": false,
 *   "f": false
 * }
 * ----
 */
@Since(version = "2.2.0")
fun isWhitespace(text: String): Boolean = native("system::IsWhitespaceStringFunctionValue")

/**
* Helper function that enables `isWhitespace` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun isWhitespace(text: Null): Boolean = false

/**
 * Checks if the `text` contains only Unicode digits. A decimal point is not a Unicode digit and returns `false`.
 *
 *
 * Note that the method does not allow for a leading sign, either positive or negative.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isAlpha` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import isAlpha from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isAlpha(null),
 *   "b": isAlpha(""),
 *   "c": isAlpha("  "),
 *   "d": isAlpha("abc"),
 *   "e": isAlpha("ab2c"),
 *   "f": isAlpha("ab-c")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": false,
 *   "f": false
 * }
 * ----
 */
@Since(version = "2.2.0")
fun isAlpha(text: String): Boolean  = native("system::IsAlphaStringFunctionValue")

/**
* Helper function that enables `isAlpha` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun isAlpha(text: Null): Boolean = false

/**
 * Checks if the `text` contains only Unicode letters or digits.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isAlphanumeric` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import isAlphanumeric from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isAlphanumeric(null),
 *   "b": isAlphanumeric(""),
 *   "c": isAlphanumeric("  "),
 *   "d": isAlphanumeric("abc"),
 *   "e": isAlphanumeric("ab c"),
 *   "f": isAlphanumeric("ab2c"),
 *   "g": isAlphanumeric("ab-c")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": false,
 *   "f": true,
 *   "g": false
 * }
 * ----
 */
@Since(version = "2.2.0")
fun isAlphanumeric(text: String): Boolean  = native("system::IsAlphaNumericStringFunctionValue")

/**
* Helper function that enables `isAlphanumeric` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun isAlphanumeric(text: Null): Boolean  = false

/**
 * Checks if the `text` contains only uppercase characters.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isUpperCase` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import isUpperCase from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isUpperCase(null),
 *   "b": isUpperCase(""),
 *   "c": isUpperCase("  "),
 *   "d": isUpperCase("ABC"),
 *   "e": isUpperCase("aBC"),
 *   "f": isUpperCase("A C"),
 *   "g": isUpperCase("A1C"),
 *   "h": isUpperCase("A/C")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": false,
 *   "f": false,
 *   "g": false,
 *   "h": false
 * }
 * ----
 */
@Since(version = "2.2.0")
fun isUpperCase(text: String): Boolean  = native("system::IsUpperCaseStringFunctionValue")

/**
* Helper function that enables `isUpperCase` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun isUpperCase(text: Null): Boolean  = false

/**
 * Checks if the `text` contains only lowercase characters.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * |===
 *
 * === Example
 *
 * This example shows how `isLowerCase` behaves with different inputs and sizes.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import isLowerCase from dw::core::Strings
 * output application/json
 * ---
 * {
 *   "a": isLowerCase(null),
 *   "b": isLowerCase(""),
 *   "c": isLowerCase("  "),
 *   "d": isLowerCase("abc"),
 *   "e": isLowerCase("aBC"),
 *   "f": isLowerCase("a c"),
 *   "g": isLowerCase("a1c"),
 *   "h": isLowerCase("a/c")
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": false,
 *   "b": false,
 *   "c": false,
 *   "d": true,
 *   "e": false,
 *   "f": false,
 *   "g": false,
 *   "h": false
 * }
 * ----
 */
@Since(version = "2.2.0")
fun isLowerCase(text: String): Boolean  = native("system::IsLowerCaseStringFunctionValue")

/**
* Helper function that enables `isLowerCase` to work with a `null` value.
*/
@Since(version = "2.2.0")
fun isLowerCase(text: Null): Boolean  = false


/**
 * Checks that the string length is no larger than the specified `maxLength`.
 * If the string's length is larger than the `maxLength`, the function cuts
 * characters from left to right, until the string length meets the length limit.
 *
 * === Parameters
 *
 * [%header, cols="1,3"]
 * |===
 * | Name | Description
 * | `text` | The input string.
 * | `maxLength` | The maximum length of the string.
 * |===
 *
 * === Example
 *
 * This example shows how `withMaxSize` behaves with different inputs and sizes.
 * Note that if `withMaxSize` is 0, the function returns an empty string. If
 * the input is `null`, the output is always `null`.
 *
 * ==== Source
 *
 * [source,DataWeave,linenums]
 * ----
 * %dw 2.0
 * import withMaxSize from dw::core::Strings
 * output application/json
 * ---
 * {
 *    a: "123" withMaxSize 10,
 *    b: "123" withMaxSize 3,
 *    c: "123" withMaxSize 2,
 *    d: "" withMaxSize 0,
 *    e: null withMaxSize 23,
 * }
 * ----
 *
 * ==== Output
 *
 * [source,JSON,linenums]
 * ----
 * {
 *   "a": "123",
 *   "b": "123",
 *   "c": "12",
 *   "d": "",
 *   "e": null
 * }
 * ----
 */
@Since(version = "2.3.0")
fun withMaxSize(text: String, maxLength: Number): String = do {
    if(maxLength < 0)
        dw::Runtime::fail("maxLength should be a positive number")
    else
        if(sizeOf(text) > maxLength)
            text[0 to maxLength - 1]
        else
            text
}

/**
* Helper function that enables `withMaxSize` to work with a `null` value.
*/
@Since(version = "2.3.0")
fun withMaxSize(text: Null, maxLength: Number): Null = null

/**
* Replaces all substrings that match a literal search string with
* a specified replacement string.
*
*
* Replacement proceeds from the beginning of the string to the end.
* For example, the result of replacing `"aa"` with `"b"` in the
* string` `"aaa"` is `"ba"`, rather than `"ab"`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to search.
* | `target` | The string to find and replace in `text`.
* | `replacement` | The replacement string.
* |===
*
* === Example
*
* This example shows how `replaceAll` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* import * from dw::core::Strings
* output application/json
* ---
* {
*     a: replaceAll("Mariano", "a" , "A"),
*     b: replaceAll("AAAA", "AAA" , "B"),
*     c: replaceAll(null, "aria" , "A"),
*     d: replaceAll("Mariano", "j" , "Test"),
* }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*    "a": "MAriAno",
*    "b": "BA",
*    "c": null,
*    "d": "Mariano"
*  }
* ----
*/
@Since(version = "2.4.0")
fun replaceAll(text: String, target: String, replacement: String): String =
    text replace target with replacement

/**
* Helper function that enables `replaceAll` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun replaceAll(text: Null, oldValue: String, newValue: String): Null =
    null

/**
* Reverses sequence of characters in a string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to reverse.
* |===
*
* === Example
*
* This example shows how `reverse` behaves with different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
*  {
*      a: reverse("Mariano"),
*      b: reverse(null),
*      c: reverse("")
*  }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "a": "onairaM",
*   "b": null,
*   "c": ""
* }
* ----
**/
@Since(version = "2.4.0")
fun reverse(text: String): String = text[-1 to 0] default ""

/**
* Helper function that enables `reverse` to work with a `null` value.
**/
@Since(version = "2.4.0")
fun reverse(text: Null): Null = null

/**
* Returns a substring that spans from the character at the
* specified `from` index to the last character before the
* `until` index.
*
*
* The characters in the substring satisfy the condition
* `from &lt;= indexOf(string) < until`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name | Description
* | `text` | The string, treated as an array of characters.
* | `from` | The lowest index to include from the character array.
* | `until` | The lowest index to exclude from the character array.
* |===
*
* === Example
*
* This example returns the substring with characters at indices
* `1` through `4` in the input string.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* var text = "hello world!"
* ---
* substring(text, 1, 5)
* ----
*
* ==== Output
*
* [source,json,linenums]
* ----
* "ello"
* ----
*/
@Since(version = "2.4.0")
fun substring(text: String, from: Number, until: Number): String = do {
    if (from < 0)
        substring(text, 0, until)
    else if (from >= until)
        ""
    else
        text[from to (until-1)] default text[from to -1] default ""
}

/**
* Helper function that enables `substring` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun substring(text: Null, from: Any, until: Any): Null = null

/**
* Splits a string into an array of substrings equal to a specified length.
*
*
* The last substring can be shorter than that length. If the length
* is greater than or equal to the length of the string to split, the
* function returns the entire string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to split.
* | `amount` | The desired length of each substring.
* |===
*
* === Example
*
* This example shows how `substringEvery` behaves when splitting an
* input string. The last returned substring is shorter than the others.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import substringEvery from dw::core::Strings
* output application/json
* ---
* substringEvery("substringEvery", 3)
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* ["sub", "str", "ing", "Eve", "ry"]
* ----
**/
@Since(version = "2.4.0")
fun substringEvery(text: String, amount: Number): Array<String> = do {
    fun accumulate(item: String, acc: {last: String, rest: Array<String>}) = do {
        if (sizeOf(acc.last) >= amount)
            {last: item, rest: acc.rest ++ [acc.last]}
        else
            {last: acc.last ++ item, rest: acc.rest}
    }
    ---
    if (amount <= 0)
        []
    else
        do{
            var reduce_result = text reduce (item, acc={last: "", rest: []}) -> accumulate(item, acc)
            ---
            if (not isEmpty(reduce_result.last))
                reduce_result.rest ++ [reduce_result.last]
            else
                reduce_result.rest
        }
}

/**
* Helper function that enables `substringEvery` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun substringEvery(text: Null, amount: Any): Null = null

/**
* Splits a string at each character where the `predicate` expression
* returns `true`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to split. The string is treated as an array of characters.
* | `predicate` | Expression that tests each character and returns a
*                 Boolean value. The expression can iterate over each
*                 character and index of the string.
* |===
*
* === Example
*
* This example splits a string where any of the specified characters
* (`"~"`, `"="`, or `"_"`) are present.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import substringBy from dw::core::Strings
* output application/json
* ---
* "hello~world=here_data-weave" substringBy $ == "~" or $ == "=" or $ == "_"
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* ["hello", "world", "here", "data-weave"]
* ----
**/
@Since(version = "2.4.0")
fun substringBy(text: String, predicate: (character: String, index: Number) -> Boolean): Array<String> = do {
    fun reductor(character, acc) = do {
        if (predicate(character, acc.idx))
            {last: "", rest: acc.rest ++ [acc.last], idx: acc.idx + 1}
        else
            {last: acc.last ++ character, rest: acc.rest, idx: acc.idx + 1}
    }
    var reduce_result = text reduce (character, acc = {last: "", rest: [], idx: 0}) -> reductor(character, acc)
    ---
    reduce_result.rest ++ [reduce_result.last]
}

/**
* Helper function that enables `substringBy` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun substringBy(text: Null, predicate: (character: Nothing, index: Nothing) -> Any): Null = null

/**
* Collapses the string into substrings of equal characters.
*
*
* Each substring contains a single character or identical characters
* that are adjacent to one another in the input string. Empty spaces
* are treated as characters.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to collapse.
* |===
*
* === Example
*
* This example shows how the function collapses characters.
* Notice that the empty space (" ") is treated as a character.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import collapse from dw::core::Strings
* output application/json
* ---
* collapse("a  b babb a")
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* ["a", "  ", "b", " ", "b", "a", "bb", " ", "a"]
* ----
**/
@Since(version = "2.4.0")
fun collapse(text: String): Array<String> = do{
    fun accumulate(item, acc) =
        if (item == acc.last)
            {last: acc.last, current: acc.current ++ item, result: acc.result}
        else if (acc.last == null)
            {last: item, current: item, result: acc.result}
        else
            {last: item, current: item, result: acc.result ++ [acc.current]}
    var reduceResult = text reduce (item, acc = {last: null, result: [], current: ""}) -> accumulate(item, acc)
    ---
    reduceResult.result ++ [reduceResult.current]
}

/**
* Helper function that enables `collapse` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun collapse(text: Null): Null = null

/**
* Returns an array of words from a string.
*
*
* Separators between words include blank spaces, new lines, and tabs.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to split into words.
* |===
*
* === Example
*
* This example divides a string by the `words` it contains.
* An `\n` represents a line break, and `\t` represents a tab.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import words from dw::core::Strings
* output application/json
* ---
* words("hello world\nhere\t\t\tdata-weave")
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* ["hello", "world", "here", "data-weave"]
* ----
**/
@Since(version = "2.4.0")
fun words(text:String): Array<String> = text splitBy /[\s]/ filter not isBlank($)

/**
* Helper function that enables `words` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun words(text:Null): Null = null

/**
* Returns an array of lines from a string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to split into lines.
* |===
*
* === Example
*
* This example divides a string into lines.
* An `\n` represents a line break.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import lines from dw::core::Strings
* output application/json
* ---
* lines("hello world\n\nhere   data-weave")
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* ["hello world", "", "here   data-weave"]
* ----
**/
@Since(version = "2.4.0")
fun lines(text:String): Array<String> = text splitBy /\r\n|\r|\n/

/**
* Helper function that enables `lines` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun lines(text:Null): Null = null

/**
* Returns characters from the beginning of a string to the
* specified number of characters in the string, for example,
* the first two characters of a string.
*
*
* If the number is equal to or greater than the number of characters
* in the string, the function returns the entire string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to process.
* | `amount` | The number of characters to return. Negative
*              numbers and `0` return an empty string. Decimals
*              are rounded down to the nearest whole number.
* |===
*
* === Example
*
* This example returns the first five characters from a string.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import first from dw::core::Strings
* output application/json
* ---
* "hello world!" first 5
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* "hello"
* ----
**/
@Since(version = "2.4.0")
fun first(text: String, amount: Number): String = substring(text, 0, amount)

/**
* Helper function that enables `first` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun first(text: Null, amount: Any): Null = null

/**
* Returns characters from the end of string to a
* specified number of characters, for example, the last
* two characters of a string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to process.
* | `amount` | The number of characters to return. Negative
*              numbers and `0` return an empty string. Decimals
*              are rounded up to the nearest whole number.
* |===
*
* === Example
*
* This example returns the last six characters from a string.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import last from dw::core::Strings
* output application/json
* ---
* "hello world!" last 6
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* "world!"
* ----
**/
@Since(version = "2.4.0")
fun last(text: String, amount: Number): String = substring(text, sizeOf(text) - amount, sizeOf(text))

/**
* Helper function that enables `last` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun last(text: Null, amount: Any): Null = null

/**
* Counts the number of matches in a string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to search for matches.
* | `pattern` | A substring to find in the text.
* |===
*
* === Example
*
* This example counts matches in a string.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import countMatches from dw::core::Strings
* output application/json
* ---
* "hello worlo!" countMatches "lo"
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* 2
* ----
**/
@Since(version = "2.4.0")
fun countMatches(text: String, pattern: String): Number = sizeOf(text find pattern)

/**
* Counts the number of times a regular expression matches text in a string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to search for matches.
* | `pattern` | The regex pattern to use in the search.
* |===
*
* === Example
*
* This example counts the vowels in a string.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import countMatches from dw::core::Strings
* output application/json
* ---
* "hello, ciao!" countMatches /[aeiou]/
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* 5
* ----
**/
@Since(version = "2.4.0")
fun countMatches(text: String, pattern: Regex): Number = sizeOf(text find pattern)

/**
* Helper function that enables `countMatches` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun countMatches(text: Null, pattern: Any): Null = null

/**
* Counts the number of times an expression that iterates through
* each character in a string returns `true`.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to which the `predicate` applies.
* | `predicate` | Expression to apply to each character in the
*                 `text` string. The expression must return a
*                 Boolean value.
* |===
*
* === Example
*
* This example counts the digits in a string.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import countCharactersBy from dw::core::Strings
* output application/json
* ---
* "42 = 11 * 2 + 20" countCharactersBy isNumeric($)
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* 7
* ----
**/
@Since(version = "2.4.0")
fun countCharactersBy(text: String, predicate: (character: String) -> Boolean): Number =
    text reduce (item, acc=0) -> if (predicate(item)) acc + 1 else acc

/**
* Helper function to make `countCharactersBy` work with a `null` value.
*/
@Since(version = "2.4.0")
fun countCharactersBy(text: Null, predicate: (character: Nothing) -> Any): Null = null

/**
* Checks whether a condition is valid for _every_ character in a string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to check.
* | `condition` | Expression that iterates through the characters in
*                 the string that it checks and returns a Boolean value.
* |===
*
* === Example
*
* This example determines whether a string is composed
* of only digits and spaces.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import everyCharacter from dw::core::Strings
* output application/json
* ---
* "12 34  56" everyCharacter $ == " " or isNumeric($)
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* true
* ----
**/
@Since(version = "2.4.0")
fun everyCharacter(text: String, condition: (character: String) -> Boolean): Boolean =
    text reduce (item, acc=true) -> condition(item) and acc

/**
* Helper function that enables `everyCharacter` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun everyCharacter(text: Null, condition: (character: Nothing) -> Any): true = true

/**
* Checks whether a condition is valid for at least one of the characters or blank spaces
* in a string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to check.
* | `condition` | Expression that iterates through the characters and spaces
*                 in the string and returns a Boolean value.
* |===
*
* === Example
*
* This example determines whether a string has any uppercase characters.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import someCharacter from dw::core::Strings
* output application/json
* ---
* "someCharacter" someCharacter isUpperCase($)
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* true
* ----
**/
@Since(version = "2.4.0")
fun someCharacter(text: String, condition: (character: String) -> Boolean): Boolean =
    text reduce (item, acc=false) -> condition(item) or acc

/**
* Helper function that enables `someCharacter` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun someCharacter(text: Null, condition: (character: Nothing) -> Any): false = false

/**
* Removes all occurrences of a specified pattern from a string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The text to remove from.
* | `toRemove` | The pattern to remove.
* |===
*
* === Example
*
* This example shows how the `remove` can be used to remove some unwanted properties.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import remove from dw::core::Strings
* output application/json
* ---
* "lazyness purity state higher-order stateful" remove "state"
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* "lazyness purity  higher-order ful"
* ----
**/
@Since(version = "2.4.0")
fun remove(text: String, toRemove: String) : String =
    text replace toRemove with ""

/**
* Helper function that enables `remove` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun remove(text: Null, toRemove: Any) : Null = null

/**
* Applies an expression to every character of a string.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `text` | The string to map.
* | `mapper` | Expression that applies to each character (`$`) or
*              index (`$$`) of the `text` string and returns a string.
* |===
*
* === Example
*
* This example redacts sensitive data from a string.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import * from dw::core::Strings
* output application/json
* ---
* { balance: ("\$234" mapString if (isNumeric($)) "~" else $) }
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* {
*   "balance": "$~~~"
* }
* ----
**/
@Since(version = "2.4.0")
fun mapString(@StreamCapable text: String, mapper: (character: String, index: Number) -> String): String = do {
    fun accumulator(item: String, acc: {index: Number, result: String}): {index: Number, result: String} =
        {index: acc.index + 1, result: acc.result ++ mapper(item, acc.index)}
    ---
    (text reduce (item, acc= {index: 0, result: ""}) -> accumulator(item, acc)).result
}

/**
* Helper function that enables `mapString` to work with a `null` value.
*/
@Since(version = "2.4.0")
fun mapString(@StreamCapable text: Null, mapper: (character: Nothing, index: Nothing) -> Any): Null = null

/**
* Returns the Hamming distance between two strings.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `a` | The first string.
* | `b` | The second string.
* |===
*
* === Example
*
* This example shows how `hammingDistance` behaves with different strings.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import hammingDistance from dw::core::Strings
* output application/json
* ---
* "holu" hammingDistance "chau"
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* 3
* ----
**/
@Since(version = "2.4.0")
fun hammingDistance(a: String, b: String): Number | Null = do {
    fun accumulator(item, acc) =
        if (item == b[acc.index])
            {index: acc.index + 1, result: acc.result}
        else
            {index: acc.index + 1, result: acc.result + 1}
    ---
    if (sizeOf(a) == sizeOf(b))
        (a reduce (item, acc = {index: 0, result: 0}) -> accumulator(item, acc)).result
    else
        null
    }

/**
* Returns the Levenshtein distance (or edit distance) between two strings.
*
* === Parameters
*
* [%header, cols="1,3"]
* |===
* | Name   | Description
* | `a` | The first string.
* | `b` | The second string.
* |===
*
* === Example
*
* This example shows how `levenshteinDistance` behaves with different strings.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* import levenshteinDistance from dw::core::Strings
* output application/json
* ---
* "kitten" levenshteinDistance "sitting"
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
* 3
* ----
**/
@Since(version = "2.4.0")
fun levenshteinDistance(a: String, b: String): Number = do{
    var sizeA = sizeOf(a)
    var sizeB = sizeOf(b)
    fun levenshteinRow(idxa: Number, idxb: Number, previous: Array<Number>, current: Array<Number>): Array<Number> = do{
        if (idxb == (sizeB + 1))
            current
        else
            do {
                var result: Number =
                    if (idxb == 0)
                        idxa
                    else if (a[idxa - 1] == b[idxb - 1])
                        previous[idxb - 1]
                    else
                        min([
                            current[idxb - 1] + 1,
                            previous[idxb - 1] + 1,
                            previous[idxb] + 1,
                        ]) default previous[idxb] + 1
                ---
                levenshteinRow(idxa, idxb + 1, previous, current << result /*TODO optimize*/)
            }
    }

    fun levenshtein(idxa: Number, previous: Array<Number>): Number =
        if (idxa == (sizeA + 1))
            previous[sizeB]
        else
            levenshtein(idxa + 1, levenshteinRow(idxa, 0, previous, []))

    ---
    levenshtein(1, 0 to sizeB)
}
