%dw 2.0
import * from scripts::modules::ApiGraphModule

fun getTypeByName(schema, name) =
    (getShapes(schema) filterTypes ['shacl:NodeShape'] filter ($.'shacl:name' == name))[0]

type ScalarKind = 'boolean' | 'float' | 'integer' | 'string' | 'ID' | 'anyType'
fun getScalarKind(shape): ScalarKind =
    shape.'raml-shapes:format'
    default shape.'shacl:datatype'[0].'@id' replace /^[^#]*#/ with ''

type ShapeKind = 'array' | 'null' | 'recursive' | 'enum' | 'scalar' | 'union' | 'interface' | 'record' | 'unknown'
fun getShapeKind(shape): ShapeKind =
    if (shape.'@type' contains 'raml-shapes:ArrayShape') 'array'
    else if (shape.'@type' contains 'raml-shapes:NilShape') 'null'
    else if (shape.'@type' contains 'raml-shapes:RecursiveShape') 'recursive'
    else if ((shape.'@type' contains 'raml-shapes:ScalarShape') and shape.'shacl:in' != null) 'enum'
    else if (shape.'@type' contains 'raml-shapes:ScalarShape') 'scalar'
    else if (shape.'@type' contains 'raml-shapes:UnionShape') 'union'
    else if ((shape.'@type' contains 'shacl:NodeShape') and shape.'raml-shapes:isAbstract' == true) 'interface'
    else if (shape.'@type' contains 'shacl:NodeShape') 'record'
    else 'unknown'

type Field = {
    name: String,
    shapeID: String,
    isOptional: Boolean
}
fun propertiesFor(schema, shape): Array<Field> = do {
    var propertyIDs = shape.'shacl:property'.'@id'
    var properties = schema.'@graph' matchingObjectsById propertyIDs
    ---
    properties map {
        name: $.'shacl:name',
        shapeID: $.'raml-shapes:range'.'@id',
        isOptional: $.'shacl:minCount' == 0
    }
}

fun getExampleWithTypename(schema, shape, maxDepth: Number) = do {
    var example = getExample(schema, shape.'@id', maxDepth)
    ---
    if (example is Object)
        { '__typename': shape.'shacl:name' } ++ example
    else
        example
}

fun getRecordExample(schema, shape, maxDepth: Number) =
    if (maxDepth > 0)
        propertiesFor(schema, shape) reduce (v, acc={}) ->
            acc ++ { (v.name): getExample(schema, v.shapeID, maxDepth) }
    else
        null

fun getScalarExample(shape) = getScalarKind(shape) match {
    case 'boolean' -> choose([true, false])
    case 'float' -> choose([123.456, 3.14159265, 1.414213562, 0.0000001])
    case 'integer' -> choose([-100, -10, -2, -1, 0, 1, 2, 10, 100])
    case 'string' -> choose(['Hello World!', 'Goodbye', 'This is some example data'])
    case 'ID' -> choose(['some-id', '#myID', 'urn:uuid:123e4567-e89b-12d3-a456-426655440000'])
    case 'anyType' -> 'Scaffolded value for $(shape."shacl:name" default "some unknown type")'
    else -> 'Scaffolded value for $(shape."shacl:name" default "some unknown type")'
}

fun getArrayExample(schema, shape, maxDepth: Number) = do {
    var arraySize = choose([[], [1], [2, 2], [3, 3, 3]])
    ---
    arraySize map (getExample(schema, shape.'raml-shapes:items'.'@id', maxDepth))
}

fun getUnionExample(schema, shape, maxDepth: Number) = do {
    // Fields with types like `[Something]` become `null | [null | Something]`
    // after AMF parses the API.
    //
    // We don't want to have a lot of dumb nulls in our examples, so we will
    // unwrap those unions.
    var unwrapped = do {
        var variantIDs = shape.'raml-shapes:anyOf'.'@id'
        var variants = schema.'@graph' matchingObjectsById variantIDs
        var nonNullVariants = variants filter (getShapeKind($) != 'null')
        ---
        if (sizeOf(nonNullVariants) == 1)
            nonNullVariants[0]
        else
            null
    }
    ---
    if (unwrapped != null)
        getExample(schema, unwrapped.'@id', maxDepth)
    else do {
        var variants = (schema.'@graph' matchingObjectsById shape.'raml-shapes:anyOf'.'@id')
        var variant = choose(variants)
        ---
        getExampleWithTypename(schema, variant, maxDepth)
    }
}

fun getInterfaceExample(schema, shape, maxDepth) = do {
    var implementations = getShapes(schema)
        filter ($.'raml-shapes:inherits' != null)
        filter ($.'raml-shapes:inherits'.'@id' contains shape."@id")
    ---
    getExampleWithTypename(schema, choose(implementations), maxDepth)
}

fun getEnumExample(schema, shape) = do {
    var list = (schema.'@graph' getObjectById shape.'shacl:in'.'@id')[0]
    var valueIDs =
        entriesOf(list)
        filter ($.key startsWith 'rdfs:')
        map $.value.'@id'
    var values = schema.'@graph' matchingObjectsById valueIDs
    ---
    choose(values.'data:value')
}

fun getExample(schema, shapeID: String, maxDepth: Number=3) = do {
    var shape = (schema.'@graph' getObjectById shapeID)[0]
    ---
    getShapeKind(shape) match {
        case 'record' -> getRecordExample(schema, shape, maxDepth-1)
        case 'scalar' -> getScalarExample(shape)
        case 'recursive' -> getExample(schema, shape.'raml-shapes:fixPoint'.'@id'[0], maxDepth)
        case 'null' -> null
        case 'array' -> getArrayExample(schema, shape, maxDepth)
        case 'union' -> getUnionExample(schema, shape, maxDepth)
        case 'interface' -> getInterfaceExample(schema, shape, maxDepth)
        case 'enum' -> getEnumExample(schema, shape)
        case 'unknown' -> 'The scaffolding process failed to produce a type for $(shapeID)'
    }
}
