%dw 2.0
import * from scripts::modules::ScaffoldingModule
import * from scripts::modules::PropertiesModule
import * from scripts::modules::ApiGraphModule
import * from scripts::asyncapi::AsyncApiModule

var salesforcepubsub = {
    prefix: "salesforce-pub-sub",
    uri: "http://www.mulesoft.org/schema/mule/salesforce-pub-sub"
} as Namespace

ns tls http://www.mulesoft.org/schema/mule/tls
ns http http://www.mulesoft.org/schema/mule/http

var serverPrefix = "salesforcepubsub.server."
var salesforcePlatformEventsConfig = "Salesforce_Platform_Events_configuration"
var supportedSecuritySchemes: Array<String> = ["userPassword", "X509", "oauth2"]
var httpListenerConfigNamePrefix = "HTTP_Listener_config"
var httpSchema = http.uri ++ " http://www.mulesoft.org/schema/mule/http/current/mule-http.xsd"
var salesforcePubSubSchema = salesforcepubsub.uri ++ " http://www.mulesoft.org/schema/mule/salesforce-pub-sub/current/mule-salesforce-pub-sub.xsd"
fun getSchemaLocation(): String = salesforcePubSubSchema ++ " " ++ httpSchema

/**
* Returns property specifications for Salesforce Platform Events protocol
*/
fun getEnvironmentFile(api, apiPath) = do {
    var servers = getServersForProtocol(api, "salesforcepubsub") filter hasOperationForServer(api, $."@id")
    ---
    getServerEnvParams(api, servers, serverPrefix)
    ++ getConnectionParams(servers, api)
}

/**
* The params username, password and securityToken are required for Salesforce Plaform Events configuration.
* We will use placeholders for these, and users will be required to complete them in the properties file.
* As Oauth Username Password is default it will be the value in case no security is set
**/
fun getConnectionParams(servers, api) = do {
    flatten(
        servers
        map ((server, indx) -> (
            chooseSecurityScheme(getSecuritySchemesForServer(api, server)) match {
                case schemeType if (schemeType ~= "OAuth 2.0") -> [
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".consumerSecret", ""),
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".consumerKey", ""),
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".callbackPath", ""),
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".authorizePath", "")
                ]
                else -> [
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".username", ""),
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".password", ""),
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".securityToken", ""),
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".consumerSecret", ""),
                    addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".consumerKey", "")
                ]
            }
        ))
    ) ++
    flatten(
        servers
        map ((server, indx) -> ( [
            addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".reconnection.failsDeployment", "true"),
            addPropertyIfNotPresent(serverPrefix ++ server."core:name" ++ ".reconnection.reconnect.count", "100")
        ]
        ))
    )
}


fun scaffoldAsyncApiConfig(api) = do {
    var servers =  getServersForProtocol(api, "salesforcepubsub") filter hasOperationForServer(api, $."@id")
    ---
    if (!isEmpty(servers)) {
        apikitEda#"salesforce-pub-sub-configs": {
            (
                servers map
                (
                    apikitEda#"salesforce-pub-sub-config" @(
                        "serverKey": $."core:name",
                        "configRef": salesforcePlatformEventsConfig ++ "_" ++ $."core:name"
                        ): {}
                )
            )
        }
    } else {}
}

fun scaffoldConfigs(api, existingConfiguration) = {(
    getServersForProtocol(api, "salesforcepubsub")
    filter hasOperationForServer(api, $."@id")
    map do {
        var schemes = getSecuritySchemesForServer(api, $)
        var securityScheme = chooseSecurityScheme(schemes)
        var configName = salesforcePlatformEventsConfig ++ "_" ++ $."core:name"
        var httpListenerConfigName = httpListenerConfigNamePrefix ++ "_" ++ $."core:name"
        ---
        scaffoldElement(
            (salesforcepubsub#"pubsub-config" @("name": configName): {
                (generateConnectionTypeTemplate($, securityScheme))
            }),
            lookupElementFromApp(
                existingConfiguration,
                (value, key, index) -> key ~= "pubsub-config" and value.@name == configName
            )
        ) ++
        if (securityScheme ~= "OAuth 2.0") {
            (generateListenerConfig(existingConfiguration, httpListenerConfigName))
        } else {}
    }
)}

fun generateListenerConfig(existingConfiguration, httpListenerConfigName) =
    scaffoldElement(
        (http#"listener-config" @("name": httpListenerConfigName): {
            http#"listener-connection" @("host": "0.0.0.0", "port": "8082"):{}
        }),
        lookupElementFromApp(
            existingConfiguration,
            (value, key, index) -> key ~= "listener-config" and value.@name == httpListenerConfigName
        )
    )

/*
* Prioritize and pick one of the supported security schemes first.
* If not, choose the first security scheme declared from the list.
*/
fun chooseSecurityScheme(schemes) =  do {
    var supportedSchemes = schemes filter (supportedSecuritySchemes contains $)
    ---
    if (!isEmpty(supportedSchemes)) (
        supportedSchemes[0]."security:type"
    )
    else (
        schemes[0]."security:type"
    )
}

/**
*  Server Connection templates
*  OAuth: Will include OAuth 2.0 configuration
*  Any other securityScheme or an empty value will generate OAuth UserPassword configuration
*/
fun generateConnectionTypeTemplate(server, securityScheme) = do {
    securityScheme match {
        case schemeType if (schemeType ~= "OAuth 2.0") ->
            (generateConnectionTemplateOauth2(server, securityScheme))
        else -> (generateConnectionTemplateBasicConnection(server, securityScheme))
    }
}

fun generateConnectionTemplateOauthUsernameAndPassword(server, securityScheme) = do {
    var username = "\${" ++ serverPrefix ++ server."core:name" ++ ".username}"
    var password = "\${" ++ serverPrefix ++ server."core:name" ++ ".password}"
    var consumerSecret = "\${" ++ serverPrefix ++ server."core:name" ++ ".consumerSecret}"
    var consumerKey = "\${" ++ serverPrefix ++ server."core:name" ++ ".consumerKey}"
    ---
    salesforcepubsub#"oauth-user-pass-connection" @(
        "username": username,
        "password": password,
        "consumerSecret": consumerSecret,
        "consumerKey": consumerKey
    ): {
        (generateSecurityTemplate(securityScheme, server))
    }
}

fun generateConnectionTemplateOauth2(server, securityScheme) = {
    salesforcepubsub#"config-with-oauth-connection": {
        (generateSecurityTemplate(securityScheme, server))
    }
}

/**
* This method generates the basic connection and is the alternative usernamePassword security scheme without OAuth
* Given we are using OAuth this template is not meant to be used unless we need it in the future.
**/
fun generateConnectionTemplateBasicConnection(server, securityScheme) = do {
    var username = "\${" ++ serverPrefix ++ server."core:name" ++ ".username}"
    var password = "\${" ++ serverPrefix ++ server."core:name" ++ ".password}"
    var securityToken = "\${" ++ serverPrefix ++ server."core:name" ++ ".securityToken}"
    ---
    salesforcepubsub#"basic-connection" @(
        "username": username,
        "password": password,
        "securityToken": securityToken
    ): {
        (generateSecurityTemplate(securityScheme, server))
    }
}

fun generateReconnection(server) = do {
    var failsDeployment = "\${" ++ serverPrefix ++ server."core:name" ++ ".reconnection.failsDeployment}"
    var reconnectCount = "\${" ++ serverPrefix ++ server."core:name" ++ ".reconnection.reconnect.count}"
    ---
    "reconnection" @("failsDeployment": failsDeployment): {
        "reconnect" @("count": reconnectCount):{}
    }}

fun generateSecurityTemplate(securityScheme, server) = do {
    var consumerSecret = "\${" ++ serverPrefix ++ server."core:name" ++ ".consumerSecret}"
    var consumerKey = "\${" ++ serverPrefix ++ server."core:name" ++ ".consumerKey}"
    var callbackPath = "\${" ++ serverPrefix ++ server."core:name" ++ ".callbackPath}"
    var authorizePath = "\${" ++ serverPrefix ++ server."core:name" ++ ".authorizePath}"
    var httpListenerConfigName = httpListenerConfigNamePrefix ++ "_" ++ server."core:name"
    ---
    (generateReconnection(server)) ++
    (securityScheme match {
        case schemeType if (schemeType ~= "OAuth 2.0") -> {
            salesforcepubsub#"oauth-authorization-code" @(
                "consumerKey": consumerKey,
                "consumerSecret": consumerSecret
            ): {},
            salesforcepubsub#"oauth-callback-config" @(
                "listenerConfig": httpListenerConfigName,
                "callbackPath": callbackPath,
                "authorizePath": authorizePath
            ): {}
        }
        case schemeType if (schemeType ~= "X509") ->
            tls#"context": {
                tls#"trust-store" @("path": "path/to/trustore"): {},
                tls#"key-store" @("path": "path/to/keystore"): {}
            }
        else -> {}
    })
}
