%dw 2.0
import fail from dw::Runtime
import firstWith from dw::core::Arrays
import * from scripts::modules::ApiGraphModule
import * from scripts::modules::PropertiesModule
import * from scripts::modules::ScaffoldingModule
import getMuleNamespaces from scripts::modules::SharedComponentsModule

var apikitEda = {
    prefix: "apikit-asyncapi",
    uri: "http://www.mulesoft.org/schema/mule/apikit-asyncapi"
} as Namespace

var unknownProtocol: String = "unknown"
var asyncApiConfigName = "asyncapi-config"
var supportedProtocols: Array<String> = ["kafka", "kafka-secure", "anypointmq", "solace", "salesforcepubsub"]

fun getAsyncApiSchemaLocation(): String = apikitEda.uri ++ " http://www.mulesoft.org/schema/mule/apikit-asyncapi/current/mule-apikit-asyncapi.xsd"

fun hasSubscribeOperations(api) = !isEmpty(getEndpoints(api) map getEndpointOperations(api, $) flatMap $ firstWith $."apiContract:method" == "subscribe")

fun hasPublishOperations(api) = !isEmpty(getEndpoints(api) map getEndpointOperations(api, $) flatMap $ firstWith $."apiContract:method" == "publish")

fun getPublishOpTypes(api) = getOperations(api) filter $."apiContract:method"? and $."apiContract:method" == "publish"

fun getSubscribeOpTypes(api) = getOperations(api) filter $."apiContract:method"? and $."apiContract:method" == "subscribe"

fun getEndpointFromPublishOpId(api, publishOpId) = getEndpoints(api) firstWith (flattenIds($."apiContract:supportedOperation") contains publishOpId)

// gets the server objects from api graph
fun getServers(api): Array = api."@graph" filterTypes ["apiContract:Server"]

// get server objects for a protocol from api graph
fun getServersForProtocol(api, protocol: String) = getServers(api) filter ((item, index) -> item."apiContract:protocol" contains protocol)

/*
 * Get unique protocol names from the api graph
 * Returns array of strings
 */
fun getUniqueSupportedProtocolNames(api): Array<String> = do {
    var servers = getServers(api)
    ---
    if (!isEmpty(servers))
    (
        servers map $."apiContract:protocol" distinctBy $ filter ((item, index) -> supportedProtocols contains item)
    )
    else
        fail("Need to specify at least one server in the AsyncAPI Spec")
}

/*
 * This acts like a validation wrapper around getUniqueSupportedProtocolNames
 * function to validate the supported protocols
 */
fun getProtocols(api): Array<String> = do {
    var supportedProtocols = getUniqueSupportedProtocolNames(api)
    ---
    if (!isEmpty(supportedProtocols))
        supportedProtocols map if ($ == "kafka-secure") "kafka" else $
    else fail("None of the protocols in the AsyncAPI spec are currently supported")
}

fun replaceInvalidCharacters(channelName) = channelName
    replace "{" with ("(")
    replace "}" with (")")
    replace "/" with ("\\")
    replace "[" with ("((")
    replace "]" with ("))")
    replace "#" with ("@")

/*
 * Returns an xml tag with a format:
 *
 * <?xml version='1.0' encoding='UTF-8'?>
 * <xsi:schemaLocation xmlns:xsi="http://www.w3.org/2001/XMLSchema-instance">
 * http://www.mulesoft.org/schema/mule/core http://www.mulesoft.org/schema/mule/core/current/mule.xsd
 * http://www.mulesoft.org/schema/mule/kafka http://www.mulesoft.org/schema/mule/kafka/current/mule-kafka.xsd
 * http://www.mulesoft.org/schema/mule/anypointmq http://www.mulesoft.org/schema/mule/anypointmq/current/mule-anypointmq.xsd
 * </xsi:schemaLocation>
 */
fun getMuleNamespacesByProtocolSchemas(schemas) =
    if (isEmpty(schemas)) fail("Protocols in the AsyncAPI Spec are currently not supported")
    else
    {
        (getMuleNamespaces(schemas))
    }

/*
 * Return all publish operations that has the matching binding for the specified protocol
*/
fun getPubOperationsWithProtocolBinding(api, protocolName, bindingName) = do {
    getPublishOpTypes(api) 
    filter $."apiBinding:binding"?
    filter !isEmpty(getFirstOpBindingForProtocol(api, $, protocolName, bindingName))
}

/*
 * For a given operation, fetch the first matching binding object that
 * matches the given protocol and binding name
*/
fun getFirstOpBindingForProtocol(api, op, protocolName, bindingName) = do {
    getBindingsForOperation(api, op)
    firstWith ((binding, index) -> (binding."apiBinding:type" contains protocolName) and binding[("apiBinding:" ++ bindingName)]?)
}

fun getDefaultValueFromSchema(api, schema) = 
    if(schema."shacl:defaultValue"?) (
    getObjectById(api."@graph", schema."shacl:defaultValue"."@id")[0]."data:value"
    )
    else if (schema."shacl:in"?) (
    getValueFromEnumList(api, schema)
    )
    else ("")

fun getValueFromEnumList(api, schema) = do {
    var enumListObj = getObjectById(api."@graph", schema."shacl:in"."@id")[0]
    ---
    getObjectById(api."@graph", enumListObj."rdfs:_1"."@id")[0]."data:value"
}

fun getSchemaFromParameter(api, param) = getObjectById(api."@graph", param."raml-shapes:schema"."@id")[0]

fun getDefaultValueFromParam(api, param) = getDefaultValueFromSchema(api, getSchemaFromParameter(api, param))

/*
 * Given a binding object, get the binding schema
 * for the specified api binding value
 * binding object example ->  {
      "@id": "#53",
      "@type": [
        "apiBinding:KafkaOperationBinding",
        "apiBinding:OperationBinding",
        "doc:DomainElement"
      ],
      "apiBinding:clientId": {
        "@id": "#54"
      },
      "apiBinding:bindingVersion": "latest",
      "apiBinding:type": "kafka"
    },
 */
fun getBindingSchema(api, bindingObj, bindingName) = getObjectById(api."@graph", bindingObj[("apiBinding:" ++ bindingName)]."@id")[0]

/*
 * Given a list of server objects, returns a list of (key): value
 * property objects for each server parameter, where 'key' is the
 * full parameter name, and 'value' is the default value assigned.
*/
fun getServerEnvParams(api, servers, paramPrefix) = do {
    var parameters = getParameters(api)
    ---
    flatten(
        servers
        filter $."apiContract:variable"?
        map ((server, indx) -> matchingObjectsById(parameters, flattenIds(server."apiContract:variable"))
            reduce (
                (serverParamObj, acc = []) ->
                acc ++ [addPropertyIfNotPresent(
                    paramPrefix ++ server."core:name" ++ "." ++ serverParamObj."core:name", getDefaultValueFromParam(api, serverParamObj))]
            )
        )
    )
}

/*
 * NOTE: We are not adding paramater placeholders for channel parameters at the moment.
 *       This method is not actively used.
 * Returns a list of (key): value
 * property objects for each endpoint/channel parameter, where 'key' is the
 * full parameter name, and 'value' is the default value assigned.
*/
fun getEndpointEnvParams(api) = do {
    var parameters = getParameters(api)
    var endpointParamObjs = getEndpoints(api)
        filter $."apiContract:parameter"?
        flatMap matchingObjectsById(parameters, flattenIds($."apiContract:parameter"))
        groupBy $."core:name"
    ---
    endpointParamObjs pluck ((value, key, index) ->
    value firstWith getDefaultValueFromParam(api, $) != "" default value[0])
    reduce (
        (param, acc = []) ->
            acc ++
            [addPropertyIfNotPresent("channel." ++ param."core:name", getDefaultValueFromParam(api, param))]
    )
}

/*
 * Scaffold a message listener flow with following steps:
 * 1) Get channels for the given operation ID and retrieve server names assigned to the channel (Only supported servers will be used)
 * 2) Use 'operationId' of the operation as the flow name OR the channel path if the former is not present
 * 3) Look up the existing flow based on the flow name
 * 4) Scaffold a new flow if element does not exist, else only add servers to existing flow
*/
fun scaffoldAsyncApiSubscribeFlow(api, existingConfiguration, servers, publishOp) = do {
    var channel = getEndpointFromPublishOpId(api, publishOp."@id")
    var supportedServers = servers filter ((item, index) -> supportedProtocols contains item."apiContract:protocol")
    var serverNames = supportedServers filter (flattenIds(channel."apiContract:server") contains $."@id") map $."core:name"
    var flowName1 = "LISTEN:" ++ (publishOp."apiContract:operationId" default "null")
    var flowName2 = "LISTEN:" ++ replaceInvalidCharacters(channel."apiContract:path")
    var existingFlow = lookupElementFromApp(existingConfiguration, (value, key, index) -> key ~= "flow" and (value.@name == flowName1 or value.@name == flowName2))
    ---
    if (isEmpty(existingFlow)) {
        flow @("name": (if(publishOp."apiContract:operationId"?) flowName1 else flowName2)): {
                apikitEda#"message-listener" @("config-ref": asyncApiConfigName, "channelName": channel."apiContract:path"): {
                    apikitEda#"servers": {
                        (
                            serverNames map apikitEda#"server" @(value: $): {}
                        )
                    }
                },
                logger @("level": "INFO", "message": "#[payload]"): {}
        }
    } else {
        (addServersToExistingFlow(existingFlow, serverNames))
    }
}

fun addServersToExistingFlow(flowElement, serverNames) = do {
    flowElement mapObject ((value, key, index) -> 
        if (key ~= "servers") {
            (key) : {
                (
                    serverNames map apikitEda#"server" @(value: $): {}
                )
            }
        } else if (value is Object) {
            (key): addServersToExistingFlow(value, serverNames)
        } else {
            (key): value
        }
    )
}

/*
 * Determines whether there is at least one subscribe operation associated
 * with the server ID
*/
fun hasSubscribeOperationForServer(api, serverId) = do {
    !isEmpty(getEndpoints(api)
    filter (flattenIds($."apiContract:server") contains serverId)
    map getEndpointOperations(api, $) flatMap $ firstWith $."apiContract:method" == "subscribe")
}

/*
 * Determines whether there is at least one publish operation associated
 * with the server ID
*/
fun hasPublishOperationForServer(api, serverId) = do {
    !isEmpty(getEndpoints(api)
    filter (flattenIds($."apiContract:server") contains serverId)
    map getEndpointOperations(api, $) flatMap $ firstWith $."apiContract:method" == "publish")
}

fun hasOperationForServer(api, serverId) = do {
    !isEmpty(getEndpoints(api)
    firstWith (
        !isEmpty($."apiContract:supportedOperation")
        and (flattenIds($."apiContract:server") contains serverId)
    ))
}
