/**
 * Copyright (c) MuleSoft, Inc. All rights reserved. http://www.mulesoft.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.md file.
 */

package com.mule.modules.ftplite;

import com.mule.modules.ftplite.exceptions.FtpLiteAuthException;
import com.mule.modules.ftplite.exceptions.FtpLiteException;
import com.mule.modules.ftplite.exceptions.FtpLiteHostException;
import com.mule.modules.ftplite.io.FtpConnectionClosingInputStream;
import org.apache.commons.net.ftp.FTP;
import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPFile;
import org.apache.commons.net.ftp.FTPReply;

import java.io.IOException;
import java.io.InputStream;
import java.net.SocketException;
import java.net.UnknownHostException;

public class FtpUtils {
    protected static FTPFile[] getFiles(FTPClient client) throws FtpLiteException {
        FTPFile[] files;
        try {
            files = client.listFiles();
        } catch (IOException e) {
            throw new FtpLiteException(e);
        }

        if (!FTPReply.isPositiveCompletion(client.getReplyCode()))
        {
            throw new FtpLiteException("Ftp error: " + client.getReplyCode(), client.getReplyCode());
        }

        return files;
    }

    protected static FTPFile getFile(FTPClient client, String filePath) throws FtpLiteException {
        FTPFile[] files;
        try {
            files = client.listFiles(filePath);
        } catch (IOException e) {
            throw new FtpLiteException(e);
        }

        if (!FTPReply.isPositiveCompletion(client.getReplyCode()))
        {
            throw new FtpLiteException("Ftp error: " + client.getReplyCode(), client.getReplyCode());
        }

        if (files.length == 1)
            return files[0];

        return null;
    }

    protected static FTPClient createClient(String hostName, String port, String passive, String userName, String password, String path) throws FtpLiteException
    {
        try {
            int remotePort = Integer.parseInt(port);
            boolean passiveConnection = Boolean.parseBoolean(passive);

            FTPClient client =  new FTPClient();

            if (remotePort > 0)
            {
                client.connect(hostName, remotePort);
            }
            else
            {
                client.connect(hostName);
            }
            if (!FTPReply.isPositiveCompletion(client.getReplyCode()))
            {
                throw new FtpLiteHostException("Ftp connect failed: " + client.getReplyCode(), client.getReplyCode());
            }
            if (!client.login(userName, password))
            {
                throw new FtpLiteAuthException("Ftp login failed: " + client.getReplyCode(), client.getReplyCode());
            }
            if (!client.setFileType(FTP.BINARY_FILE_TYPE))
            {
                throw new FtpLiteException("Ftp error. Couldn't set BINARY transfer type: " + client.getReplyCode(), client.getReplyCode());
            }
            if (path!=null && !client.changeWorkingDirectory(path)) {
                throw new FtpLiteException("Ftp error. Couldn't set change working directory: " + client.getReplyCode(), client.getReplyCode());
            }
            if (passiveConnection) {
                client.enterLocalPassiveMode();
            }
            return client;
        }
        catch (SocketException e) {
            throw new FtpLiteException(e);

        } catch (UnknownHostException e) {
            throw new FtpLiteHostException(e);

        } catch (IOException e) {
            throw new FtpLiteException(e);

        }
    }

    public static void releaseClient(FTPClient client) {
        try {
            client.quit();
        } catch (IOException e) {
            // We do not care at all if there are issues when closing the connection
        }
    }

    protected static InputStream getFileContent(FTPClient client, String filePath) {
        try {
            InputStream result = client.retrieveFileStream(filePath);
            return new FtpConnectionClosingInputStream(client, result);
        } catch (IOException e) {
            return null;
        }
    }

    public static void uploadStream(FTPClient client, String fileName, InputStream content) {
        try {
            if (!client.storeFile(fileName, content)) {
                throw new FtpLiteException("Ftp error. Couldn't upload file: " + client.getReplyCode(), client.getReplyCode());
            }
        } catch (IOException e) {
            throw new FtpLiteException(e);
        }

    }
}
