/**
 * Copyright (c) MuleSoft, Inc. All rights reserved. http://www.mulesoft.com
 *
 * The software in this package is published under the terms of the CPAL v1.0
 * license, a copy of which has been included with this distribution in the
 * LICENSE.md file.
 */
/**
 * This file was automatically generated by the Mule Development Kit
 */
package com.mule.modules.ftplite;

import com.mule.modules.ftplite.exceptions.FtpLiteException;

import org.apache.commons.net.ftp.FTPClient;
import org.apache.commons.net.ftp.FTPFile;
import org.mule.api.annotations.Connector;
import org.mule.api.annotations.Processor;
import org.mule.api.annotations.param.Default;
import org.mule.api.annotations.param.Optional;
import org.mule.util.FilenameUtils;

import java.io.InputStream;

/**
 * FTP Light connector. It takes credentials in all calls. Opens and closes a connection on each
 * call to the server. If you need a persistent connection to an FTP server, you would be better
 * served by using the FTP transport in Mule.
 *
 * @author MuleSoft, Inc.
 */
@Connector(name="ftplite", schemaVersion="1.0", friendlyName="LiteFtp", minMuleVersion="3.4")
public class FtpLiteConnector
{
	/**
	 * The default value for the port property in all connector operations
	 */
	private final static String STANDARD_FTP_PORT = "21";
	
	/**
	 * The default value for the passive property in all connector operations
	 */
	private final static String DEFAULT_PASSIVE_MODE_ENABLED = "true";
	
	/**
	 * The default folder name
	 */
	private final static String DEFAULT_FOLDER_PATH = "/";

    /**
     * Get all folder and files in a Path, it defaults to "/" when path is null
     *
     * {@sample.xml ../../../doc/FtpLite-connector.xml.sample ftplite:get-folder}
     *
     * @param hostName The FTP host's name to connect to
     * @param userName The user name to use to login
     * @param password The password to use to login
     * @param port the port the FTP service is listening on
     * @param path the path to the folder to list
     * @param passive use passive mode on the connection
     * @return an array of FTPFile's in the path specified
     * @throws FtpLiteException when there is an error in the FTP connection
     */
    @Processor
    public FTPFile[] getFolder(
    		String hostName,
    		String userName,
    		String password,
    		@Optional @Default(STANDARD_FTP_PORT) String port,
    		@Optional @Default(DEFAULT_FOLDER_PATH) String path,
    		@Optional @Default(DEFAULT_PASSIVE_MODE_ENABLED) String passive) throws FtpLiteException
    {
    	FTPClient client = FtpUtils.createClient(hostName, port, passive, userName, password, path);
    	FTPFile[] result = FtpUtils.getFiles(client);
        FtpUtils.releaseClient(client);
    	return result;
    }

	/**
     * Get a single file's information. filePath must be the full file's path
     *
     * {@sample.xml ../../../doc/FtpLite-connector.xml.sample ftplite:get-file}
     *
     * @param hostName The FTP host's name to connect to
     * @param userName The user name to use to login
     * @param password The password to use to login
     * @param port the port the FTP service is listening on
     * @param filePath the path to the folder to list
     * @param passive use passive mode on the connection
     * @return an FTPFile entry with the file's information 
     * @throws FtpLiteException when there is IO error in the FTP connection
     */
    @Processor
    public FTPFile getFile(
    		String hostName,
    		String userName,
    		String password,
    		@Optional @Default(STANDARD_FTP_PORT) String port,
    		String filePath,
    		@Optional @Default(DEFAULT_PASSIVE_MODE_ENABLED) String passive) throws FtpLiteException
    {
        String path, fileName;
        path = FilenameUtils.getFullPathNoEndSeparator(filePath);
        fileName = FilenameUtils.getName(filePath);
        FTPClient client = FtpUtils.createClient(hostName, port, passive, userName, password, path);
    	FTPFile file = FtpUtils.getFile(client, fileName);
        FtpUtils.releaseClient(client);
    	return file;
    }
    
    /**
     * Get a single file's content as a stream
     *
     * {@sample.xml ../../../doc/FtpLite-connector.xml.sample ftplite:get-file-content}
     *
     * @param hostName The FTP host's name to connect to
     * @param userName The user name to use to login
     * @param password The password to use to login
     * @param port the port the FTP service is listening on
     * @param filePath the path to the folder to list
     * @param passive use passive mode on the connection
     * @return an FTPFile entry with the file's information 
     * @throws FtpLiteException when there is an error in the FTP connection
     */
    @Processor
    public InputStream getFileContent(
    		String hostName,
    		String userName,
    		String password,
    		@Optional @Default(STANDARD_FTP_PORT) String port,
    		String filePath,
    		@Optional @Default(DEFAULT_PASSIVE_MODE_ENABLED) String passive) throws FtpLiteException
    {
    	FTPClient client = FtpUtils.createClient(hostName, port, passive, userName, password, null);
    	InputStream result = FtpUtils.getFileContent(client, filePath);
    	return result;
    }

    /**
     * Get a single file's content as a stream
     *
     * {@sample.xml ../../../doc/FtpLite-connector.xml.sample ftplite:get-file-content}
     *
     * @param hostName The FTP host's name to connect to
     * @param userName The user name to use to login
     * @param password The password to use to login
     * @param port the port the FTP service is listening on
     * @param filePath the path to the folder to store the file in
     * @param fileName the name of the file to store
     * @param content an InputStream with the content to store in the file
     * @param passive use passive mode on the connection
     * @return an FTPFile entry with the file's information
     * @throws FtpLiteException when there is an error in the FTP connection
     */
    @Processor
    public FTPFile uploadStream(
            String hostName,
            String userName,
            String password,
            @Optional @Default(STANDARD_FTP_PORT) String port,
            String filePath,
            String fileName,
            @Optional @Default("#[payload]") InputStream content,
            @Optional @Default(DEFAULT_PASSIVE_MODE_ENABLED) String passive) throws FtpLiteException
    {
        FTPClient client = FtpUtils.createClient(hostName, port, passive, userName, password, filePath);
        FtpUtils.uploadStream(client, fileName, content);
        FTPFile file = FtpUtils.getFile(client, fileName);
        FtpUtils.releaseClient(client);
        return file;
    }
}
