package org.mule.modules.wsdl.header;

import javax.xml.namespace.QName;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPHeader;

import org.apache.commons.lang.Validate;
import org.mule.modules.wsdl.WSDLInvocation;

/**
 * SalesForce header builder for the {@link WSDLInvocation WSDL Invoker}.
 *
 * @author martin.paoloni@mulesoft.com
 */
public class SalesForceHeaderBuilder implements SOAPHeaderBuilder {

	/** The SalesForce Session ID. */
	private String salesForceSessionId;

	/**
	 * Constructs a {@link SalesForceHeaderBuilder WSSE SalesForce Header
	 * Builder} to use with the {@link WSDLInvocation WSDL Invoker}.
	 *
	 * @param salesForceSessionId
	 *            The SalesForce Session ID. Cannot be null or empty.
	 */
	public SalesForceHeaderBuilder(String salesForceSessionId) {
		Validate.notEmpty(salesForceSessionId,
				"The SalesForce Session ID cannot be null or empty.");
		this.salesForceSessionId = salesForceSessionId;
	}

	@Override
	public void build(SOAPHeader header, WSDLInvocation wsdlInvocation)
			throws SOAPHeaderException {
		String wsdlNamespace = wsdlInvocation.getWsdlNamespace();
		String headerQNamePrefix = wsdlInvocation.getHeaderQNamePrefix();
		Validate.notEmpty(wsdlNamespace,
				"WSDLInvocation must have wsdlNamespace defined.");
		Validate.notEmpty(headerQNamePrefix,
				"WSDLInvocation must have headerQNamePrefix defined.");

		// add element AllowFieldTruncationHeader
		SOAPElement allowFieldTruncationHeader;
		try {
			allowFieldTruncationHeader = header.addChildElement(new QName(
					wsdlNamespace, "AllowFieldTruncationHeader",
					headerQNamePrefix));
			allowFieldTruncationHeader.addChildElement(
					new QName(wsdlNamespace, "allowFieldTruncation",
							headerQNamePrefix)).addTextNode("1");

			// add element DebuggingHeader
			SOAPElement debuggingHeader = header.addChildElement(new QName(
					wsdlNamespace, "DebuggingHeader", headerQNamePrefix));
			SOAPElement categories = debuggingHeader.addChildElement(new QName(
					wsdlNamespace, "categories", headerQNamePrefix));
			categories.addChildElement(
					new QName(wsdlNamespace, "category", headerQNamePrefix))
					.addTextNode("Apex_code");
			categories.addChildElement(
					new QName(wsdlNamespace, "level", headerQNamePrefix))
					.addTextNode("Info");
			debuggingHeader.addChildElement(
					new QName(wsdlNamespace, "debugLevel", headerQNamePrefix))
					.addTextNode("Detail");

			// add element CallOptions
			SOAPElement callOptions = header.addChildElement(new QName(
					wsdlNamespace, "CallOptions", headerQNamePrefix));
			callOptions.addChildElement(
					new QName(wsdlNamespace, "client", headerQNamePrefix))
					.addTextNode("32.0");

			// add element SessionHeader
			SOAPElement sessionHeader = header.addChildElement(new QName(
					wsdlNamespace, "SessionHeader", headerQNamePrefix));
			sessionHeader.addChildElement(
					new QName(wsdlNamespace, "sessionId", headerQNamePrefix))
					.addTextNode(salesForceSessionId);

		} catch (SOAPException e) {
			throw new SOAPHeaderException(e);
		}
	}
}
