/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */
package org.mule.modules.wsdl;

import static org.apache.commons.lang.StringUtils.isBlank;
import static org.apache.commons.lang.StringUtils.isNotBlank;
import static org.apache.commons.lang.StringUtils.join;

import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import javax.wsdl.Binding;
import javax.wsdl.BindingOperation;
import javax.wsdl.Definition;
import javax.wsdl.Message;
import javax.wsdl.Operation;
import javax.wsdl.Output;
import javax.wsdl.Part;
import javax.wsdl.WSDLException;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLReader;
import javax.xml.namespace.QName;
import javax.xml.transform.TransformerException;

import org.apache.commons.httpclient.util.URIUtil;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang.WordUtils;
import org.apache.cxf.catalog.CatalogWSDLLocator;
import org.mule.common.metadata.DefaultMetaData;
import org.mule.common.metadata.DefaultMetaDataKey;
import org.mule.common.metadata.DefaultSimpleMetaDataModel;
import org.mule.common.metadata.DefaultUnknownMetaDataModel;
import org.mule.common.metadata.MetaData;
import org.mule.common.metadata.MetaDataKey;
import org.mule.common.metadata.XmlMetaDataModel;
import org.mule.common.metadata.builder.DefaultMetaDataBuilder;
import org.mule.common.metadata.builder.XmlMetaDataBuilder;
import org.mule.common.metadata.datatype.DataType;

import com.google.common.collect.ImmutableMap;

/**
 * Helper class that parses and processes the {@link MetaData} in WSDL files.
 *
 * @author martin.paoloni@mulesoft.com
 */
public class WSDLMetadata {

    /** The {@link List} of {@link MetaDataKey}s. */
    private final List<MetaDataKey> metaDataKeys = new ArrayList<MetaDataKey>();

    /** Map with DataTypes that is used internally for the creation of the MetaData objects. */
    private final Map<String, DataType> types = ImmutableMap.<String, DataType> builder().put("string", DataType.STRING).put("boolean", DataType.BOOLEAN)
            .put("date", DataType.DATE).put("decimal", DataType.DECIMAL).put("byte", DataType.BYTE).put("unsignedByte", DataType.BYTE).put("dateTime", DataType.DATE_TIME)
            .put("int", DataType.INTEGER).put("integer", DataType.INTEGER).put("unsignedInt", DataType.INTEGER).put("short", DataType.INTEGER)
            .put("unsignedShort", DataType.INTEGER).put("long", DataType.LONG).put("unsignedLong", DataType.LONG).put("double", DataType.DOUBLE).build();

    /** Maps {@link MetaDataKey MetaDataKeys} with the corresponding {@link URL} of their WSDLs. */
    private final Map<MetaDataKey, URL> metaDataKeysUrls = new HashMap<MetaDataKey, URL>();

    /**
     * Add metadata from the wsdl URLs list received.
     *
     * @param wsdlURLs
     *            An {@link Iterable} with the WSDL {@link URL URLs}.
     * @param category
     *            A category name. Can be the simple name of the class that references this
     * @param suffix
     *            A {@link String} to append at the end of the Metadata Name. Can be null.
     *
     * @throws WSDLMetadataException
     *             If there is a problem parsing the Metadata.
     */
    public void addMetadata(Iterable<URL> wsdlURLs, String category, String suffix) throws WSDLMetadataException {

        for (URL wsdlURL : wsdlURLs) {
            try {
                // check if this conversion is necessary.
                URI wsdlURI = new URI(wsdlURL.toString());
                // This works for URLs that finish with (wsdlname).wsdl
                String name = URIUtil.getName(wsdlURI.toString());
                // This is good for URLs that finish with /(wsdlname)/?wsdl
                if (isBlank(name)) {
                    String[] path = URIUtil.getPath(wsdlURI.toString()).split("/");
                    name = path[path.length - 1];
                }
                String wsdlName = FilenameUtils.getBaseName(name);
                WSDLFactory factory = WSDLFactory.newInstance();
                WSDLReader wsdlReader = factory.newWSDLReader();
                Definition wsdlDefinition = wsdlReader.readWSDL(new CatalogWSDLLocator(wsdlURI.toString()));
                Map<QName, Binding> bindings = getBindingsFromWsdlDefinition(wsdlDefinition);

                for (QName qName : bindings.keySet()) {
                    Binding binding = bindings.get(qName);
                    for (Object bindingOperation : binding.getBindingOperations()) {
                        Operation operation = ((BindingOperation) bindingOperation).getOperation();
                        String operationName = operation.getName();
                        String nonNullSuffix = isNotBlank(suffix) ? suffix : "";

                        String metaDataId = join(new Object[] {
                                wsdlName,
                                operationName }, '#');
                        String metaDataName = join(new Object[] {
                                WordUtils.capitalizeFully(wsdlName.replace('_', ' ')),
                                WordUtils.capitalizeFully(operationName.replace('_', ' ')) + nonNullSuffix }, " - ");
                        DefaultMetaDataKey metaDataKey = new DefaultMetaDataKey(metaDataId, metaDataName);
                        metaDataKey.setCategory(category);
                        metaDataKeys.add(metaDataKey);
                        metaDataKeysUrls.put(metaDataKey, wsdlURL);
                    }
                }
            } catch (URISyntaxException e) {
                throw new WSDLMetadataException(e);
            } catch (WSDLException e) {
                throw new WSDLMetadataException(e);
            }
        }
    }

    /**
     * Gets the list of MetaDataKeys. <br>
     * The composition of a {@link MetaDataKey} is a {@link String} with the following format: <code>{WSDL name}#{Operation name}</code>.
     *
     * @return The List of {@link MetaDataKey MetaDataKeys}.
     */
    public List<MetaDataKey> getMetaDataKeys() {
        return metaDataKeys;
    }

    /**
     * Returns the input {@link MetaData} for the given {@link MetaDataKey key}.
     *
     * @param key
     *            The {@link MetaDataKey key}.
     * @return The input {@link MetaData} object.
     * @throws WSDLMetadataException 
     *             If there is an issue reading or parsing the WSDL.
     */
    public MetaData getInputMetaData(final MetaDataKey key) throws WSDLMetadataException {
        // XSD should be extracted from WSDL. Sample:
        // https://github.com/mulesoft/Mule-Tooling/blob/develop/org.mule.tooling.webservice.consumer/src/org/mule/tooling/webservice/consumer/datasense/RetrieveWsdlMetadataRunnable.java

        URL wsdlUrl = metaDataKeysUrls.get(key);

        String wsdlAndOperation[] = key.getId().split("#");
        final String operationName = wsdlAndOperation[1];
        try {
            WSDLFactory factory = WSDLFactory.newInstance();
            WSDLReader wsdlReader = factory.newWSDLReader();
            Definition wsdlDefinition = wsdlReader.readWSDL(wsdlUrl.toString());
            
            final List<String> schemas = WSDLSchemaUtils.getSchemas(wsdlDefinition);
            Operation operation = getOperationFromWsdl(wsdlDefinition, operationName);
            Message message = getInputMessage(operation);

            Part part = getMessagePart(message);
            return createMetaData(schemas, part, wsdlUrl);
        } catch (TransformerException e) {
            throw new WSDLMetadataException("Problem reading schemas from wsdl definition", e);
        } catch (WSDLException e) {
            throw new WSDLMetadataException("Problem reading schemas from wsdl definition", e);
        }
    }

    /**
     * Returns the output {@link MetaData} for the given {@link MetaDataKey key}.
     *
     * @param key
     *            The {@link MetaDataKey key}.
     * @return The output {@link MetaData} object.
     * @throws WSDLMetadataException 
     *             If there is an issue reading or parsing the WSDL.
     */
    public MetaData getOutputMetaData(final MetaDataKey key) throws WSDLMetadataException {
        // XSD should be extracted from WSDL. Sample:
        // https://github.com/mulesoft/Mule-Tooling/blob/develop/org.mule.tooling.webservice.consumer/src/org/mule/tooling/webservice/consumer/datasense/RetrieveWsdlMetadataRunnable.java

        URL wsdlUrl = metaDataKeysUrls.get(key);

        String wsdlAndOperation[] = key.getId().split("#");
        final String operationName = wsdlAndOperation[1];

        try {
            WSDLFactory factory = WSDLFactory.newInstance();
            WSDLReader wsdlReader = factory.newWSDLReader();
            Definition wsdlDefinition = wsdlReader.readWSDL(wsdlUrl.toString());

            final List<String> schemas = WSDLSchemaUtils.getSchemas(wsdlDefinition);
            Operation operation = getOperationFromWsdl(wsdlDefinition, operationName);
            Message message = getOutputMessage(operation);

            Part part = getMessagePart(message);
            MetaData metadata = null;
            if (part != null) {
                metadata = createMetaData(schemas, part, wsdlUrl);
            } else {
                metadata = new DefaultMetaData(new DefaultUnknownMetaDataModel());
            }
            return metadata;
        } catch (TransformerException e) {
            throw new WSDLMetadataException("Problem reading schemas from wsdl definition", e);
        } catch (WSDLException e) {
            throw new WSDLMetadataException("Problem reading schemas from wsdl definition", e);
        }
    }

    /**
     * Gets the Input {@link Message} from the given {@link Operation}.
     *
     * @param operation
     *            An {@link Operation}.
     * @return The Input {@link Message}.
     */
    private Message getInputMessage(Operation operation) {
        return operation.getInput().getMessage();
    }

    /**
     * Gets the Output {@link Message} from the given {@link Operation}.
     *
     * @param operation
     *            An {@link Operation}.
     * @return The Output {@link Message}.
     */
    private Message getOutputMessage(Operation operation) {
        Output output = operation.getOutput();
        return output != null ? output.getMessage() : null;
    }

    /**
     * Gets the {@link Operation} from the WSDL {@link Definition} that matches the given {@link String operation name}.
     *
     * @param wsdlDefinition
     *            The WSDL {@link Definition} from which the {@link Operation} should be gotten.
     * @param operationName
     *            A {@link String} representing the operation name required.
     * @return The requested {@link Operation}.
     */
    private Operation getOperationFromWsdl(Definition wsdlDefinition, final String operationName) {
        Map<QName, Binding> bindings = getBindingsFromWsdlDefinition(wsdlDefinition);
        Binding binding = bindings.get(bindings.keySet().toArray()[0]);
        BindingOperation bindingOperation = binding.getBindingOperation(operationName, null, null);
        Operation operation = bindingOperation.getOperation();
        return operation;
    }

    /**
     * Gets all {@link Binding bindings} for the given WSDL {@link Definition}.
     *
     * @param wsdlDefinition
     *            A WSDL {@link Definition}.
     * @return A {@link Map}, containing {@link QName QNames} as keys, and their corresponding WSDL {@link Definition definitions} as values.
     */
    private Map<QName, Binding> getBindingsFromWsdlDefinition(Definition wsdlDefinition) {
        @SuppressWarnings("unchecked")
        Map<QName, Binding> bindings = wsdlDefinition.getBindings();
        return bindings;
    }

    /**
     * Gets the first {@link Part} of a {@link Message}.
     *
     * @param message
     *            The {@link Message}.
     * @return The requested {@link Part}, or <code>null</code> if the {@link Message} is also <code>null</code>.
     */
    private Part getMessagePart(Message message) {
        if (message == null) {
            return null;
        }
        Map<?, ?> parts = message.getParts();
        if (!parts.isEmpty()) {
            Object firstValueKey = parts.keySet().toArray()[0];
            return (Part) parts.get(firstValueKey);
        }
        return null;
    }

    /**
     * Given a {@link List} of {@link String Strings} with the schemas, a {@link Part} and the WSDL {@link URL}, creates the {@link MetaData} object.
     *
     * @param schemas
     *            The {@link List} of {@link String Strings} with the schemas.
     * @param part
     *            The {@link Part} of a {@link Message}.
     * @param wsdlUrl
     *            The WSDL {@link URL}.
     * @return The requested {@link MetaData} object.
     */
    private MetaData createMetaData(List<String> schemas, Part part, URL wsdlUrl) {
        if (part != null) {
            if (part.getElementName() != null) {
                final QName elementName = part.getElementName();
                final XmlMetaDataBuilder<?> createXmlObject = new DefaultMetaDataBuilder().createXmlObject(elementName);
                for (String schema : schemas) {
                    createXmlObject.addSchemaStringList(schema);
                }
                createXmlObject.setEncoding(Charset.defaultCharset());
                createXmlObject.setExample("");
                XmlMetaDataModel model = createXmlObject.build();
                // if the operation has parameters create the XmlMetaDataModel else DataSense should
                // render Payload - Unknown
                if (!model.getFields().isEmpty()) {
                    return new DefaultMetaData(model);
                }
            } else if (part.getTypeName() != null) {
                DataType dataType = getDataTypeFromTypeName(part);
                DefaultSimpleMetaDataModel defaultSimpleMetaDataModel = new DefaultSimpleMetaDataModel(dataType);
                return new DefaultMetaData(defaultSimpleMetaDataModel);
            }
        }
        return new DefaultMetaData(new DefaultUnknownMetaDataModel());
    }

    /**
     * Gets the {@link DataType} corresponding to a WSDL {@link Part}.
     *
     * @param part
     *            The {@link Part}.
     * @return The corresponding {@link DataType}.
     */
    private DataType getDataTypeFromTypeName(Part part) {
        String localPart = part.getTypeName().getLocalPart();
        DataType dataType = types.get(localPart);
        return dataType != null ? dataType : DataType.STRING;
    }
}
