/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */
package org.mule.modules.wsdl.runtime;

import java.io.IOException;
import java.util.Collections;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPHeader;
import javax.xml.soap.SOAPMessage;
import javax.xml.soap.SOAPPart;
import javax.xml.stream.XMLStreamReader;
import javax.xml.transform.TransformerException;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.Dispatch;
import javax.xml.ws.Service;
import javax.xml.ws.soap.SOAPBinding;

import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.mule.modules.wsdl.runtime.header.HeaderBuilder;
import org.mule.modules.wsdl.runtime.header.SoapHeaderException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;
import org.xml.sax.SAXException;

/**
 * Helper class that invokes SOAP web services.
 *
 * @author martin.paoloni@mulesoft.com
 */
public class SoapClient {

    public static final String XMLSOAP_ORG_SOAP_ENCODING_NAMESPACE = "http://schemas.xmlsoap.org/soap/encoding/";
    private static final Logger logger = LoggerFactory.getLogger(SoapClient.class);
    private static final String HEADER_PREFIX = "headerPrefix";
    private static final String BODY_PREFIX = "bodyPrefix";
    private static final String CONNECTION_TIMEOUT = "com.sun.xml.internal.ws.connect.timeout";
    private static final String READ_TIMEOUT = "com.sun.xml.internal.ws.request.timeout";

    private final ServiceDefinition serviceDefinition;

    /**
     * The {@link HeaderBuilder} implementation. Optional.
     */
    private HeaderBuilder soapHeaderBuilder;

    /**
     * The parameters to be appended to the header.
     */
    private Map<String, String> headerParams;

    private Integer connectionTimeout;

    private Integer readTimeout;

    private SoapClient(@NotNull final ServiceDefinition service) {
        this.serviceDefinition = service;
    }

    @NotNull
    public static SoapClient create(@NotNull final ServiceDefinition service) {
        return new SoapClient(service);
    }

    /**
     * Invokes a web serviceDefinition.
     *
     * @param payload        A {link {@link XMLStreamReader reader} with the payload message to send.
     * @param callDefinition SOAP operation invocation definition
     * @return A {link {@link XMLStreamReader reader} with the output message received.
     * @throws SoapCallException Unexpected error during serviceDefinition call
     */
    @Nullable
    public XMLStreamReader invoke(@NotNull final CallDefinition callDefinition, @Nullable final XMLStreamReader payload) throws SoapCallException {

        logger.debug("Service Definition for Invocation: {}", serviceDefinition);
        logger.debug("SOAP call to endpoint: {}", callDefinition);

        XMLStreamReader result = null;
        try {

            // Create dispatcher client ...
            final String operationName = callDefinition.getOperationName();
            final String endpointPath = callDefinition.getEndpointPath();
            final Dispatch<SOAPMessage> dispatch = buildMessageDispatch(endpointPath, operationName);

            // Build request ...
            final SOAPMessage soapRequest = buildSoapRequest(payload, operationName, dispatch);

            // Invoke the endpoint synchronously
            final SOAPMessage soapResponse = dispatch.invoke(soapRequest);
            logger.debug("Client call successful.");

            // If the serviceDefinition responds a 202 status, the payload will be null.
            // Therefore soapResponse is null. See org.apache.cxf.endpoint.ClientImpl,line 609.
            if (soapResponse != null) {

                final SOAPBodyElement sourceContent = (SOAPBodyElement) soapResponse.getSOAPBody().getChildElements().next();

                // Add the namespaces from the response Envelope tag to the result
                // tag - so it can be parsed successfully by DataMapper
                final SOAPPart soapPart = soapRequest.getSOAPPart();
                final SOAPEnvelope soapEnvelope = soapPart.getEnvelope();

                result = XmlConverterUtils.soapResponseToXmlStream(soapResponse, sourceContent, soapEnvelope);
            }
        } catch (Exception e) {
            logger.warn("Error during web serviceDefinition invocation", e);
            throw SoapCallException.createCallException(e);
        }

        return result;
    }

    @NotNull
    private Dispatch<SOAPMessage> buildMessageDispatch(@NotNull final String endpointPath, @NotNull final String operationName) {

        // QNames for serviceDefinition as defined in wsdl.
        final String namespace = this.serviceDefinition.getNamespace();
        final String operation = this.serviceDefinition.getServiceName();
        final QName serviceQName = new QName(namespace, operation);

        // QName for Port As defined in wsdl.
        final String portName = this.serviceDefinition.getPortName();
        final QName portQName = new QName(namespace, portName);

        // Endpoint Address
        final String baseEndpoint = this.serviceDefinition.getBaseEndpoint();
        final String endpointAddress = baseEndpoint + endpointPath;

        // Create a dynamic Service instance
        final Service service = Service.create(serviceQName);

        // Add a port to the Service
        service.addPort(portQName, SOAPBinding.SOAP11HTTP_BINDING, endpointAddress);

        // Create a dispatch instance
        Dispatch<SOAPMessage> dispatch = service.createDispatch(portQName, SOAPMessage.class, Service.Mode.MESSAGE);

        // Optionally Configure RequestContext to send SOAPAction HTTP Header
        final Map<String, Object> rc = dispatch.getRequestContext();
        rc.put(BindingProvider.SOAPACTION_USE_PROPERTY, Boolean.TRUE);
        rc.put(BindingProvider.SOAPACTION_URI_PROPERTY, operationName);

        // Set connection and read timeout
        final Integer ctxTimeout = getConnectionTimeout();
        if (ctxTimeout != null) {
            rc.put(CONNECTION_TIMEOUT, ctxTimeout);
            logger.debug("Setting timeout to {}", ctxTimeout);
        }

        final Integer rTimeout = getReadTimeout();
        if (rTimeout != null) {
            rc.put(READ_TIMEOUT, rTimeout);
            logger.debug("Setting readTime to {}", rTimeout);
        }
        return dispatch;
    }

    /**
     * Build SOAP Request based on the provided parameters
     *
     * @param payload       Full payload to be sent without the operation
     * @param operationName Operation name to be invoked.
     * @param dispatch      Dispatch object used for the invocation.
     * @return SOAPMessage ready to be sent.
     */
    @NotNull
    private SOAPMessage buildSoapRequest(@Nullable XMLStreamReader payload, @NotNull final String operationName, @NotNull final Dispatch<SOAPMessage> dispatch)
            throws SOAPException, SoapHeaderException, TransformerException, ParserConfigurationException, SAXException, IOException {

        // Obtain a pre-configured SAAJ MessageFactory
        final SOAPBinding binding = (SOAPBinding) dispatch.getBinding();
        final MessageFactory msgFactory = binding.getMessageFactory();

        // Create SOAPMessage Request
        final SOAPMessage result = msgFactory.createMessage();

        SOAPPart part = result.getSOAPPart();

        // Gets the elements SOAPEnvelope, header and body.
        SOAPEnvelope env = part.getEnvelope();
        env.setEncodingStyle(XMLSOAP_ORG_SOAP_ENCODING_NAMESPACE);

        final String namespace = this.serviceDefinition.getNamespace();
        env.addNamespaceDeclaration(BODY_PREFIX, namespace + operationName + "/");
        env.addNamespaceDeclaration(HEADER_PREFIX, namespace);

        // Request Header
        SOAPHeader header = env.getHeader();
        if (soapHeaderBuilder != null) {
            soapHeaderBuilder.build(header, serviceDefinition);
        }

        final Map<String, String> params = getHeaderParams();
        final String headerPrefix = this.serviceDefinition.getHeaderPrefix();
        for (final String key : params.keySet()) {
            final QName qname = new QName(namespace, key, headerPrefix);
            final SOAPElement soapElement = header.addChildElement(qname);
            final String headerName = params.get(key);
            soapElement.addTextNode(headerName);
        }

        // Compose the soap:Body payload
        String payloadStr;
        if (payload != null) {
            payloadStr = XmlConverterUtils.xmlStreamToStr(payload);
        } else {

            // @Todo: This seems to be unnecessary, review if this can be removed ...
            // handle the case of NullPayload - technically this should only happen when the method has no payload param
            final String substring = namespace.substring(0, namespace.length() - 1);
            payloadStr = "<ns0:" + substring + " xmlns:ns0=\"" + substring + "\"/>";
        }
        logger.debug("Payload to send: {}", payloadStr);

        // @Todo: There is a unnecessary transformation from XMLStreamReader -> String -> DOM. This code must be reviewed.
        final Document document = XmlConverterUtils.convertStringToDocument(payloadStr);
        final SOAPBody body = env.getBody();
        body.addDocument(document);
        result.saveChanges();

        return result;
    }

    /**
     * Getter method for connection timeout.
     *
     * @return the connectionTimeout
     */
    public Integer getConnectionTimeout() {
        return connectionTimeout;
    }

    /**
     * Setter method for connectionTimeout.
     *
     * @param connectionTimeout the connectionTimeout to set
     */
    public void setConnectionTimeout(int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
    }

    /**
     * Getter method for readTimeout
     *
     * @return the readTimeout
     */
    public Integer getReadTimeout() {
        return readTimeout;
    }

    /**
     * Setter method for readTimeout.
     *
     * @param readTimeout the readTimeout to set
     */
    public void setReadTimeout(Integer readTimeout) {
        this.readTimeout = readTimeout;
    }

    @NotNull
    public Map<String, String> getHeaderParams() {
        return this.headerParams != null ? this.headerParams : Collections.<String, String>emptyMap();
    }

    /**
     * Sets the parameters to be appended to the header.
     *
     * @param headerParams the header parameters to set.
     */
    public void setHeaderParams(@NotNull final Map<String, String> headerParams) {
        this.headerParams = headerParams;
    }

    /**
     * Sets the {@link HeaderBuilder} implementation.
     *
     * @param soapHeaderBuilder the {@link HeaderBuilder} implementation.
     */
    public void setSoapHeaderBuilder(HeaderBuilder soapHeaderBuilder) {
        this.soapHeaderBuilder = soapHeaderBuilder;
    }
}
