package org.mule.modules.wsdl.runtime;

import static org.apache.commons.lang.StringUtils.join;

import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

/**
 * It represents the datasense key containing the metadata for performing a SOAP Call operation.
 */
public class CallDefinition {

    private static final Logger logger = LoggerFactory.getLogger(SoapClient.class);
    public static final String SEPARATOR = "||";

    @NotNull
    private final String endpointPath;
    @NotNull
    private final String operationName;

    public CallDefinition(@NotNull final String endpointPath, @NotNull final String operationName) {
        this.endpointPath = endpointPath;
        this.operationName = operationName;
    }

    @NotNull
    public static CallDefinition parseDatasenseKey(@NotNull final String datasenseKey) throws SoapCallException {

        // Is a valid format ?
        logger.debug("Parsing datasense key:" + datasenseKey);
        if (StringUtils.isEmpty(datasenseKey) || !datasenseKey.contains(SEPARATOR)) {
            throw SoapCallException.createMetadataInvocationException(datasenseKey);
        }

        // Let's parse it and extract their parts ...
        final String split[] = StringUtils.split(datasenseKey, SEPARATOR);
        final String endpointPath = split[0];
        final String operationName = split[1];

        if (StringUtils.isEmpty(endpointPath)) {
            throw SoapCallException.createMetadataInvocationException(datasenseKey);
        }

        if (StringUtils.isEmpty(operationName)) {
            throw SoapCallException.createMetadataInvocationException(datasenseKey);
        }
        return new CallDefinition(endpointPath, operationName);

    }

    @NotNull
    public String asString() {
        return join(new Object[] {
                endpointPath,
                operationName }, SEPARATOR);
    }

    @NotNull
    public String getEndpointPath() {
        return endpointPath;
    }

    @NotNull
    public String getOperationName() {
        return operationName;
    }

    @Override
    public String toString() {
        return "CallDefinition{" + "endpointPath='" + endpointPath + '\'' + ", operationName='" + operationName + '\'' + '}';
    }
}
