/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */
package org.mule.modules.wsdl.datasense;

import java.io.File;
import java.io.StringWriter;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.wsdl.Definition;
import javax.wsdl.Import;
import javax.wsdl.Types;
import javax.wsdl.extensions.schema.Schema;
import javax.wsdl.extensions.schema.SchemaImport;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;

import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Methods to manage schemas from a WSDL definition
 * <p/>
 * Shamelessly stolen from:
 * https://github.com/mulesoft/Mule-Tooling/blob/develop/org.mule.tooling.webservice.consumer/src/org/mule/tooling/webservice/consumer/utils/WSDLSchemaUtils.java
 */
class SchemaUtils {

    /**
     * Private constructor. This is an utility class.
     */
    private SchemaUtils() {
    }

    @SuppressWarnings("unchecked")
    @NotNull
    public static List<String> getSchemas(@NotNull final Definition wsdlDefinition) throws TransformerException {
        final Map<String, String> wsdlNamespaces = wsdlDefinition.getNamespaces();
        final List<String> schemas = new ArrayList<String>();
        final List<Types> typesList = new ArrayList<Types>();
        extractWsdlTypes(wsdlDefinition, typesList);
        for (final Types types : typesList) {
            for (final Object o : types.getExtensibilityElements()) {
                if (o instanceof Schema) {
                    schemas.addAll(resolveSchema(wsdlNamespaces, (Schema) o));
                }
            }
        }

        // Allow importing types from other wsdl
        for (final Object wsdlImportList : wsdlDefinition.getImports().values()) {
            final List<Import> importList = (List<Import>) wsdlImportList;
            for (final Import wsdlImport : importList) {
                schemas.addAll(getSchemas(wsdlImport.getDefinition()));
            }
        }

        return schemas;
    }

    @NotNull
    public static List<String> resolveSchema(final Map<String, String> wsdlNamespaces, final Schema schema) throws TransformerException {
        final List<String> schemas = new ArrayList<String>();
        fixPrefix(wsdlNamespaces, schema);
        fixSchemaLocations(schema);
        final String flatSchema = schemaToString(schema);
        schemas.add(flatSchema);
        // STUDIO-5814: generates an issue adding duplicated schemas
        return schemas;
    }

    /**
     * Extracts the "Types" definition from a WSDL and recursively from all the imports. The types are added to the typesList argument.
     */
    private static void extractWsdlTypes(@NotNull final Definition wsdlDefinition, @NotNull final List<Types> typesList) {
        // Add current types definition if present
        if (wsdlDefinition.getTypes() != null) {
            typesList.add(wsdlDefinition.getTypes());
        }
    }

    private static void fixPrefix(final Map<String, String> wsdlNamespaces, final Schema schema) {
        for (final Map.Entry<String, String> entry : wsdlNamespaces.entrySet()) {
            final boolean isDefault = StringUtils.isEmpty(entry.getKey());
            final boolean containNamespace = schema.getElement().hasAttribute("xmlns:" + entry.getKey());
            if (!isDefault && !containNamespace) {
                schema.getElement().setAttribute("xmlns:" + entry.getKey(), entry.getValue());
            }
        }
    }

    @SuppressWarnings("unchecked")
    private static void fixSchemaLocations(final Schema schema) {
        // fix imports schemaLocation in pojo
        final String basePath = getBasePath(schema.getDocumentBaseURI());
        final Map<String, Vector<SchemaImport>> oldImports = schema.getImports();
        final Collection<Vector<SchemaImport>> values = oldImports.values();
        if (!values.isEmpty()) {
            setSchemaLocationUris(basePath, values);
            // fix imports schemaLocation in dom
            fixImportSchemaLocationInDom(schema, basePath);
        }
    }

    private static void setSchemaLocationUris(final String basePath, final Collection<? extends List<SchemaImport>> values) {
        for (final List<SchemaImport> schemaImports : values) {
            for (final SchemaImport schemaImport : schemaImports) {
                final String schemaLocationURI = schemaImport.getSchemaLocationURI();
                if (schemaLocationURI != null && !schemaLocationURI.startsWith(basePath) && !schemaLocationURI.startsWith("http")) {
                    schemaImport.setSchemaLocationURI(basePath + schemaLocationURI);
                }
            }
        }
    }

    private static void fixImportSchemaLocationInDom(@NotNull final Schema schema, @NotNull final String basePath) {
        final NodeList children = schema.getElement().getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            final Node item = children.item(i);
            if ("import".equals(item.getLocalName())) {
                final NamedNodeMap attributes = item.getAttributes();
                final Node namedItem = attributes.getNamedItem("schemaLocation");
                if (namedItem != null) {
                    final String schemaLocation = namedItem.getNodeValue();
                    if (!schemaLocation.startsWith(basePath) && !schemaLocation.startsWith("http")) {
                        namedItem.setNodeValue(basePath + schemaLocation);
                    }
                }
            }
        }
    }

    private static String getBasePath(final String documentURI) {
        final File document = new File(documentURI);
        if (document.isDirectory()) {
            return documentURI;
        }

        final String fileName = document.getName();
        final int fileNameIndex = documentURI.lastIndexOf(fileName);
        if (fileNameIndex == -1) {
            return documentURI;
        }

        return documentURI.substring(0, fileNameIndex);
    }

    private static String schemaToString(final Schema schema) throws TransformerException {
        final Element element = schema.getElement();
        return elementToString(element);
    }

    private static String elementToString(final Element element) throws TransformerException {
        final StringWriter writer = new StringWriter();
        final Transformer transformer = TransformerFactory.newInstance().newTransformer();
        transformer.transform(new DOMSource(element), new StreamResult(writer));
        return writer.toString();
    }
}