package org.mule.modules.wsdl.runtime.header;

import javax.xml.namespace.QName;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPHeader;

import org.apache.commons.lang.Validate;
import org.jetbrains.annotations.NotNull;
import org.mule.modules.wsdl.runtime.ServiceDefinition;
import org.mule.modules.wsdl.runtime.SoapClient;

/**
 * WSSE Security header builder for the {@link SoapClient WSDL Invoker}.
 *
 * @author martin.paoloni@mulesoft.com
 */
public class WsseSecurityHeaderBuilder implements HeaderBuilder {

    public static final String WSDL_SECURITY_UTILITY_NAMESPACE = "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-utility-1.0.xsd";
    public static final String WSDL_SECURITY_NAMESPACE = "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-wssecurity-secext-1.0.xsd";

    private final String username;

    private final String password;

    /**
     * Constructs a {@link WsseSecurityHeaderBuilder WSSE Security Header Builder} to use with the {@link SoapClient}.
     *
     * @param username
     *            The user name. Cannot be null or empty.
     * @param password
     *            The password. Cannot be null or empty.
     */
    public WsseSecurityHeaderBuilder(@NotNull final String username, @NotNull final String password) {
        Validate.notNull(username, "The user name cannot be null or empty.");
        Validate.notNull(password, "The password cannot be null or empty.");
        this.username = username;
        this.password = password;
    }

    @Override
    public void build(@NotNull final SOAPHeader header, @NotNull final ServiceDefinition soapClient) throws SoapHeaderException {
        try {
            final SOAPElement security = header.addChildElement("Security", "wsse", WSDL_SECURITY_NAMESPACE);

            final SOAPElement usernameToken = security.addChildElement("UsernameToken", "wsse");
            usernameToken.addAttribute(new QName("xmlns:wsu"), WSDL_SECURITY_UTILITY_NAMESPACE);

            final SOAPElement usernameElement = usernameToken.addChildElement("Username", "wsse");
            usernameElement.addTextNode(username);

            final SOAPElement passwordElement = usernameToken.addChildElement("Password", "wsse");
            passwordElement.addAttribute(QName.valueOf("Type"), "http://docs.oasis-open.org/wss/2004/01/oasis-200401-wss-username-token-profile-1.0#PasswordText");
            passwordElement.addTextNode(password);

        } catch (final SOAPException e) {
            throw new SoapHeaderException(e);
        }
    }
}
