/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */
package org.mule.modules.wsdl.datasense;

import com.google.common.base.Function;
import com.google.common.base.MoreObjects;
import com.google.common.collect.ImmutableMap;
import com.google.common.collect.Iterables;
import com.google.common.collect.Maps;
import org.apache.commons.httpclient.util.URIUtil;
import org.apache.commons.io.FilenameUtils;
import org.apache.commons.lang3.StringUtils;
import org.apache.cxf.catalog.CatalogWSDLLocator;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.mule.common.metadata.*;
import org.mule.common.metadata.builder.XmlMetaDataBuilder;
import org.mule.common.metadata.datatype.DataType;
import org.mule.devkit.api.metadata.ComposedMetaDataKey;
import org.mule.modules.wsdl.datasense.binding.BindingHelper;
import org.mule.modules.wsdl.datasense.binding.DefaultBindingHelper;
import org.mule.modules.wsdl.metadataModel.ConcreteTypeXmlMetadataBuilder;

import javax.wsdl.*;
import javax.wsdl.factory.WSDLFactory;
import javax.wsdl.xml.WSDLReader;
import javax.xml.namespace.QName;
import javax.xml.transform.TransformerException;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.*;

import static org.apache.commons.lang3.StringUtils.isBlank;
import static org.apache.commons.lang3.StringUtils.isNotBlank;
import static org.mule.modules.wsdl.datasense.SchemaUtils.*;
import static org.mule.modules.wsdl.runtime.CallDefinition.SEPARATOR;

/**
 * Helper class that parses and processes the {@link MetaData} in WSDL files.
 *
 * @author martin.paoloni@mulesoft.com
 */
public class WsdlDatasenseParser {

    /**
     * Map with DataTypes that is used internally for the creation of the MetaData objects.
     */
    private static final Map<String, DataType> types;
    private static final WSDLFactory factory;

    static {
        final ImmutableMap.Builder<String, DataType> builder = ImmutableMap.builder();
        builder.put("string", DataType.STRING);
        builder.put("boolean", DataType.BOOLEAN);
        builder.put("date", DataType.DATE);
        builder.put("decimal", DataType.DECIMAL);
        builder.put("byte", DataType.BYTE);
        builder.put("unsignedByte", DataType.BYTE);
        builder.put("dateTime", DataType.DATE_TIME);
        builder.put("int", DataType.INTEGER);
        builder.put("integer", DataType.INTEGER);
        builder.put("unsignedInt", DataType.INTEGER);
        builder.put("short", DataType.INTEGER);
        builder.put("unsignedShort", DataType.INTEGER);
        builder.put("long", DataType.LONG);
        builder.put("unsignedLong", DataType.LONG);
        builder.put("double", DataType.DOUBLE);
        types = builder.build();

        try {
            factory = WSDLFactory.newInstance();
        } catch (WSDLException e) {
            throw new ExceptionInInitializerError(e);
        }

    }

    /**
     * The {@link List} of {@link MetaDataKey}s.
     */
    private final List<ComposedMetaDataKey> metaDataKeys = new ArrayList<ComposedMetaDataKey>();
    /**
     * Maps {@link MetaDataKey MetaDataKeys} with the corresponding {@link URL} of their WSDLs.
     */
    private final Map<String, URL> metaDataKeysUrls = new HashMap<String, URL>();

    /**
     * Add metadata from the wsdl URLs list received.
     *
     * @param wsdlURLs An {@link Iterable} with the WSDL {@link URL URLs}.
     * @param category A category name. Can be the simple name of the class that references this
     * @param suffix   A {@link String} to append at the end of the Metadata Name. Can be null.
     * @throws WsdlDatasenseException If there is a problem parsing the Metadata.
     */
    public void addMetadata(@NotNull final Iterable<URL> wsdlURLs, @NotNull final String category, @Nullable final String suffix) throws WsdlDatasenseException {
        addMetadata(wsdlURLs, category, suffix, new DefaultBindingHelper());
    }

    /**
     * Add metadata from the wsdl URLs list received.
     *
     * @param wsdlURLs An {@link Iterable} with the WSDL {@link URL URLs}.
     * @param category A category name. Can be the simple name of the class that references this
     * @param suffix   A {@link String} to append at the end of the Metadata Name. Can be null.
     * @param helper   A {@link BindingHelper} to determine if the metadata has a third level with the concrete types supported by the operation.
     * @throws WsdlDatasenseException If there is a problem parsing the Metadata.
     */
    public void addMetadata(@NotNull final Iterable<URL> wsdlURLs, @NotNull final String category, @Nullable final String suffix, BindingHelper helper)
            throws WsdlDatasenseException {
        final String nonNullSuffix = isNotBlank(suffix) ? " " + suffix : "";

        for (final URL wsdlURL : wsdlURLs) {
            try {
                // check if this conversion is necessary.
                final URI wsdlURI = new URI(wsdlURL.toString());
                // This works for URLs that finish with (wsdlname).wsdl
                String name = URIUtil.getName(wsdlURI.toString());

                // This is good for URLs that finish with /(wsdlname)/?wsdl
                if (isBlank(name)) {
                    final String[] path = URIUtil.getPath(wsdlURI.toString()).split("/");
                    name = path[path.length - 1];
                }
                final String wsdlName = FilenameUtils.getBaseName(name);
                final String firstLevelLabel = makeReadable(wsdlName) + nonNullSuffix;
                final Definition wsdlDefinition = factory.newWSDLReader().readWSDL(new CatalogWSDLLocator(wsdlURI.toString()));
                final Map<QName, Binding> bindings = getBindingsFromWsdlDefinition(wsdlDefinition);
                helper.addMetadataForBinding(metaDataKeys, metaDataKeysUrls, wsdlDefinition, wsdlName, firstLevelLabel, bindings, category);
            } catch (final URISyntaxException e) {
                throw new WsdlDatasenseException(e);
            } catch (final WSDLException e) {
                throw new WsdlDatasenseException(e);
            }
        }
    }

    /**
     * Gets the list of MetaDataKeys. <br>
     * The composition of a {@link MetaDataKey} is a {@link String} with the following format: <code>{WSDL name}||{Operation name}</code>.
     *
     * @return The List of {@link MetaDataKey MetaDataKeys}.
     */
    @NotNull
    public List<ComposedMetaDataKey> getMetaDataKeys() {
        return metaDataKeys;
    }

    /**
     * Returns the input {@link MetaData} for the given {@link MetaDataKey key}.
     *
     * @param key The {@link MetaDataKey key}.
     * @return A map from element name to the input {@link MetaData} object.
     * @throws WsdlDatasenseException If there is an issue reading or parsing the WSDL.
     */
    @SuppressWarnings("unchecked")
    @NotNull
    public Map<String, MetaData> getInputMetaData(final ComposedMetaDataKey key) throws WsdlDatasenseException {
        // XSD should be extracted from WSDL. Sample:
        // https://github.com/mulesoft/Mule-Tooling/blob/develop/org.mule.tooling.webservice.consumer/src/org/mule/tooling/webservice/consumer/datasense/RetrieveWsdlMetadataRunnable.java

        final URL wsdlUrl = metaDataKeysUrls.get(key.getId());

        final String wsdlAndOperation[] = StringUtils.split(key.getId(), SEPARATOR);
        final String operationName = wsdlAndOperation[1];
        try {
            final WSDLFactory factory = WSDLFactory.newInstance();
            final WSDLReader wsdlReader = factory.newWSDLReader();
            final Definition wsdlDefinition = wsdlReader.readWSDL(wsdlUrl.toString());

            final List<String> schemas = getSchemas(wsdlDefinition);
            final Operation operation = getOperationFromWsdl(wsdlDefinition, operationName);
            final Message message = getInputMessage(operation);

            Map<String, Part> parts = message.getParts();
            return getStringMetaDataMap(wsdlAndOperation, schemas, parts);

        } catch (final TransformerException e) {
            throw new WsdlDatasenseException("Problem reading schemas from wsdl definition", e);
        } catch (final WSDLException e) {
            throw new WsdlDatasenseException("Problem reading schemas from wsdl definition", e);
        }
    }

    private Map<String, MetaData> getStringMetaDataMap(final String[] wsdlAndOperation, final List<String> schemas, Map<String, Part> parts) {
        return Maps.transformValues(MoreObjects.firstNonNull(parts, ImmutableMap.<String, Part>of()), new Function<Part, MetaData>() {

            @Override
            public MetaData apply(@Nullable final Part part) {
                return createMetaData(schemas, part, wsdlAndOperation);
            }
        });
    }

    /**
     * Returns the output {@link MetaData} for the given {@link MetaDataKey key} .
     *
     * @param key The {@link MetaDataKey key}.
     * @return A map from element name to the output {@link MetaData} object.
     * @throws WsdlDatasenseException If there is an issue reading or parsing the WSDL.
     */
    @SuppressWarnings("unchecked")
    @NotNull
    public Map<String, MetaData> getOutputMetaData(final ComposedMetaDataKey key) throws WsdlDatasenseException {
        // XSD should be extracted from WSDL. Sample:
        // https://github.com/mulesoft/Mule-Tooling/blob/develop/org.mule.tooling.webservice.consumer/src/org/mule/tooling/webservice/consumer/datasense/RetrieveWsdlMetadataRunnable.java

        final URL wsdlUrl = metaDataKeysUrls.get(key.getId());

        final String wsdlAndOperation[] = StringUtils.split(key.getId(), SEPARATOR);
        final String operationName = wsdlAndOperation[1];

        try {
            final WSDLFactory factory = WSDLFactory.newInstance();
            final WSDLReader wsdlReader = factory.newWSDLReader();
            final Definition wsdlDefinition = wsdlReader.readWSDL(wsdlUrl.toString());

            final List<String> schemas = getSchemas(wsdlDefinition);
            final Operation operation = getOperationFromWsdl(wsdlDefinition, operationName);
            final Message message = getOutputMessage(operation);

            if (message != null && message.getParts() != null) {
                Map<String, Part> parts = message.getParts();
                return getStringMetaDataMap(wsdlAndOperation, schemas, parts);
            } else {
                return ImmutableMap.of();
            }
        } catch (final TransformerException e) {
            throw new WsdlDatasenseException("Problem reading schemas from wsdl definition", e);
        } catch (final WSDLException e) {
            throw new WsdlDatasenseException("Problem reading schemas from wsdl definition", e);
        }
    }

    /**
     * Gets the Input {@link Message} from the given {@link Operation}.
     *
     * @param operation An {@link Operation}.
     * @return The Input {@link Message}.
     */
    private Message getInputMessage(final Operation operation) {
        return operation.getInput().getMessage();
    }

    /**
     * Gets the Output {@link Message} from the given {@link Operation}.
     *
     * @param operation An {@link Operation}.
     * @return The Output {@link Message}.
     */
    @Nullable
    private Message getOutputMessage(final Operation operation) {
        final Output output = operation.getOutput();
        return output != null ? output.getMessage() : null;
    }

    /**
     * Gets the {@link Operation} from the WSDL {@link Definition} that matches the given {@link String operation name}.
     *
     * @param wsdlDefinition The WSDL {@link Definition} from which the {@link Operation} should be gotten.
     * @param operationName  A {@link String} representing the operation name required.
     * @return The requested {@link Operation}.
     */
    private Operation getOperationFromWsdl(final Definition wsdlDefinition, final String operationName) {
        final Map<QName, Binding> bindings = getBindingsFromWsdlDefinition(wsdlDefinition);

        final Set<QName> qNames = bindings.keySet();
        final Binding binding = bindings.get(qNames.iterator().next());
        final BindingOperation bindingOperation = binding.getBindingOperation(operationName, null, null);
        return bindingOperation.getOperation();
    }

    /**
     * Gets all {@link Binding bindings} for the given WSDL {@link Definition}.
     *
     * @param wsdlDefinition A WSDL {@link Definition}.
     * @return A {@link Map}, containing {@link QName QNames} as keys, and their corresponding WSDL {@link Definition definitions} as values.
     */
    @SuppressWarnings("unchecked")
    private Map<QName, Binding> getBindingsFromWsdlDefinition(final Definition wsdlDefinition) {
        return wsdlDefinition.getBindings();
    }

    /**
     * Given a {@link List} of {@link String Strings} with the schemas, a {@link Part} and the WSDL {@link URL}, creates the {@link MetaData} object.
     *
     * @param schemas The {@link List} of {@link String Strings} with the schemas.
     * @param part    The {@link Part} of a {@link Message}.
     * @return The requested {@link MetaData} object.
     */
    @NotNull
    private MetaData createMetaData(@NotNull final List<String> schemas, @Nullable final Part part, String wsdlAndOperation[]) {
        if (part != null) {
            if (part.getElementName() != null) {
                final QName elementName = part.getElementName();

                final XmlMetaDataBuilder<?> createXmlObject = new ConcreteTypeXmlMetadataBuilder(elementName)
                        .withConcreteType(Iterables.get(Arrays.asList(wsdlAndOperation), 2, null));

                for (final String schema : schemas) {
                    createXmlObject.addSchemaStringList(schema);
                }
                createXmlObject.setEncoding(Charset.defaultCharset());
                createXmlObject.setExample("");

                final XmlMetaDataModel model = createXmlObject.build();
                /*
                 * if the operation has parameters create the XmlMetaDataModel else DataSense should render Payload - Unknown
                 */
                if (!model.getFields().isEmpty()) {
                    return new DefaultMetaData(model);
                }
            } else if (part.getTypeName() != null) {
                final DataType dataType = getDataTypeFromTypeName(part);
                final DefaultSimpleMetaDataModel defaultSimpleMetaDataModel = new DefaultSimpleMetaDataModel(dataType);
                return new DefaultMetaData(defaultSimpleMetaDataModel);
            }
        }
        return new DefaultMetaData(new DefaultUnknownMetaDataModel());
    }

    /**
     * Gets the {@link DataType} corresponding to a WSDL {@link Part}.
     *
     * @param part The {@link Part}.
     * @return The corresponding {@link DataType}.
     */
    private DataType getDataTypeFromTypeName(final Part part) {
        final String localPart = part.getTypeName().getLocalPart();
        final DataType dataType = types.get(localPart);
        return dataType != null ? dataType : DataType.STRING;
    }
}
