/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */
package org.mule.modules.wsdl.datasense;

import static org.apache.commons.lang3.StringUtils.join;

import java.io.File;
import java.io.StringWriter;
import java.net.URI;
import java.net.URISyntaxException;
import java.net.URL;
import java.nio.charset.Charset;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collection;
import java.util.List;
import java.util.Map;
import java.util.Vector;

import javax.wsdl.Definition;
import javax.wsdl.Import;
import javax.wsdl.Types;
import javax.wsdl.WSDLException;
import javax.wsdl.extensions.schema.Schema;
import javax.wsdl.extensions.schema.SchemaImport;
import javax.wsdl.factory.WSDLFactory;
import javax.xml.XMLConstants;
import javax.xml.namespace.QName;
import javax.xml.transform.Transformer;
import javax.xml.transform.TransformerException;
import javax.xml.transform.TransformerFactory;
import javax.xml.transform.dom.DOMSource;
import javax.xml.transform.stream.StreamResult;
import org.apache.commons.lang.ArrayUtils;
import org.apache.commons.lang.StringUtils;
import org.apache.commons.lang3.text.WordUtils;
import org.apache.cxf.catalog.CatalogWSDLLocator;
import org.apache.xmlbeans.SchemaGlobalElement;
import org.apache.xmlbeans.SchemaProperty;
import org.apache.xmlbeans.SchemaType;
import org.apache.xmlbeans.SchemaTypeSystem;
import org.apache.xmlbeans.XmlException;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.mule.common.metadata.MetaDataGenerationException;
import org.mule.common.metadata.SchemaProvider;
import org.mule.common.metadata.StringBasedSchemaProvider;
import org.mule.common.metadata.util.XmlSchemaUtils;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

import com.google.common.base.Predicate;
import com.google.common.collect.Iterables;
import com.google.common.collect.Lists;

/**
 * Methods to manage schemas from a WSDL definition. Shamelessly stolen from:
 * https://github.com/mulesoft/Mule-Tooling/blob/develop/org.mule.tooling.webservice.consumer/src/org/mule/tooling/webservice/consumer/utils/WSDLSchemaUtils.java
 */
public class SchemaUtils {

    public static final int THIRD_LEVEL_COMPONENT_NUMBER = 2;
    public static final String DEFINITION_SEPARATOR = "-";
    private static final Predicate<String> NON_EMPTY_STRING = new Predicate<String>() {

        @Override
        public boolean apply(@Nullable final String input) {
            return StringUtils.isNotBlank(input);
        }
    };

    /**
     * Private constructor. This is an utility class.
     */
    private SchemaUtils() {
    }

    @SuppressWarnings("unchecked")
    @NotNull
    public static List<String> getSchemas(@NotNull final Definition wsdlDefinition) throws TransformerException {
        final Map<String, String> wsdlNamespaces = wsdlDefinition.getNamespaces();
        final List<String> schemas = new ArrayList<String>();
        final List<Types> typesList = new ArrayList<Types>();
        extractWsdlTypes(wsdlDefinition, typesList);
        for (final Types types : typesList) {
            for (final Object o : types.getExtensibilityElements()) {
                if (o instanceof Schema) {
                    schemas.addAll(resolveSchema(wsdlNamespaces, (Schema) o));
                }
            }
        }

        // Allow importing types from other wsdl
        for (final Object wsdlImportList : wsdlDefinition.getImports().values()) {
            final List<Import> importList = (List<Import>) wsdlImportList;
            for (final Import wsdlImport : importList) {
                schemas.addAll(getSchemas(wsdlImport.getDefinition()));
            }
        }

        return schemas;
    }

    @NotNull
    private static List<String> resolveSchema(final Map<String, String> wsdlNamespaces, final Schema schema) throws TransformerException {
        final List<String> schemas = new ArrayList<String>();
        fixPrefix(wsdlNamespaces, schema);
        fixSchemaLocations(schema);
        final String flatSchema = schemaToString(schema);
        schemas.add(flatSchema);
        // STUDIO-5814: generates an issue adding duplicated schemas
        return schemas;
    }

    /**
     * Extracts the "Types" definition from a WSDL and recursively from all the imports. The types are added to the typesList argument.
     */
    private static void extractWsdlTypes(@NotNull final Definition wsdlDefinition, @NotNull final List<Types> typesList) {
        // Add current types definition if present
        if (wsdlDefinition.getTypes() != null) {
            typesList.add(wsdlDefinition.getTypes());
        }
    }

    private static void fixPrefix(final Map<String, String> wsdlNamespaces, final Schema schema) {
        for (final Map.Entry<String, String> entry : wsdlNamespaces.entrySet()) {
            final boolean isDefault = StringUtils.isEmpty(entry.getKey());
            final boolean containNamespace = schema.getElement().hasAttribute("xmlns:" + entry.getKey());
            if (!isDefault && !containNamespace) {
                schema.getElement().setAttribute("xmlns:" + entry.getKey(), entry.getValue());
            }
        }
    }

    @SuppressWarnings("unchecked")
    private static void fixSchemaLocations(final Schema schema) {
        // fix imports schemaLocation in pojo
        final String basePath = getBasePath(schema.getDocumentBaseURI());
        final Map<String, Vector<SchemaImport>> oldImports = schema.getImports();
        final Collection<Vector<SchemaImport>> values = oldImports.values();
        if (!values.isEmpty()) {
            setSchemaLocationUris(basePath, values);
            // fix imports schemaLocation in dom
            fixImportSchemaLocationInDom(schema, basePath);
        }
    }

    private static void setSchemaLocationUris(final String basePath, final Collection<? extends List<SchemaImport>> values) {
        for (final List<SchemaImport> schemaImports : values) {
            for (final SchemaImport schemaImport : schemaImports) {
                final String schemaLocationURI = schemaImport.getSchemaLocationURI();
                if (schemaLocationURI != null && !schemaLocationURI.startsWith(basePath) && !schemaLocationURI.startsWith("http")) {
                    schemaImport.setSchemaLocationURI(basePath + schemaLocationURI);
                }
            }
        }
    }

    private static void fixImportSchemaLocationInDom(@NotNull final Schema schema, @NotNull final String basePath) {
        final NodeList children = schema.getElement().getChildNodes();
        for (int i = 0; i < children.getLength(); i++) {
            final Node item = children.item(i);
            if ("import".equals(item.getLocalName())) {
                final NamedNodeMap attributes = item.getAttributes();
                final Node namedItem = attributes.getNamedItem("schemaLocation");
                if (namedItem != null) {
                    final String schemaLocation = namedItem.getNodeValue();
                    if (!schemaLocation.startsWith(basePath) && !schemaLocation.startsWith("http")) {
                        namedItem.setNodeValue(basePath + schemaLocation);
                    }
                }
            }
        }
    }

    private static String getBasePath(final String documentURI) {
        final File document = new File(documentURI);
        if (document.isDirectory()) {
            return documentURI;
        }

        final String fileName = document.getName();
        final int fileNameIndex = documentURI.lastIndexOf(fileName);
        if (fileNameIndex == -1) {
            return documentURI;
        }

        return documentURI.substring(0, fileNameIndex);
    }

    private static String schemaToString(final Schema schema) throws TransformerException {
        return elementToString(schema.getElement());
    }

    private static String elementToString(final Element element) throws TransformerException {
        final StringWriter writer = new StringWriter();
        final TransformerFactory transformerFactory = TransformerFactory.newInstance("com.sun.org.apache.xalan.internal.xsltc.trax.TransformerFactoryImpl", null);
        transformerFactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_DTD, "");
        transformerFactory.setAttribute(XMLConstants.ACCESS_EXTERNAL_STYLESHEET, "");
        final Transformer transformer = transformerFactory.newTransformer();
        transformer.transform(new DOMSource(element), new StreamResult(writer));
        return writer.toString();
    }

    public static List<SchemaType> getExtendingTypes(List<String> listOfschemas, URL sourceUrl, QName rootElementName, Charset encoding) {
        try {
            SchemaProvider schemas = new StringBasedSchemaProvider(listOfschemas, encoding, sourceUrl);

            List<SchemaType> returnedTypes = new ArrayList<SchemaType>();

            SchemaTypeSystem schemaTypeLoader = XmlSchemaUtils.getSchemaTypeSystem(listOfschemas, null);
            SchemaType[] listOfElements = schemaTypeLoader.globalTypes();
            SchemaGlobalElement rootElement = schemas.findRootElement(rootElementName);
            SchemaType type = rootElement.getType();
            final SchemaProperty[] properties = type.getProperties();
            for (SchemaProperty property : properties) {

                final SchemaType propertyType = property.getType();

                if (!hasSimpleContentOnly(propertyType)) {
                    Iterables.concat(returnedTypes, Iterables.filter(Arrays.asList(listOfElements), new Predicate<SchemaType>() {

                        @Override
                        public boolean apply(SchemaType input) {
                            return input.getBaseType().getName().equals(propertyType.getName());
                        }
                    }));
                    for (SchemaType schType : listOfElements) {
                        if (schType.getBaseType().getName().equals(propertyType.getName())) {
                            returnedTypes.add(schType);
                        }
                    }
                }
            }
            return returnedTypes;
        } catch (XmlException e) {
            throw new MetaDataGenerationException(e);
        }
    }

    public static List<String> getExtendingTypeNames(@NotNull final Iterable<URL> wsdlURLs, URL sourceUrl, QName elName, Charset encoding)
            throws WSDLException, WsdlDatasenseException {
        WSDLFactory factory = WSDLFactory.newInstance();
        List<String> returnedTypeList = new ArrayList<String>();
        for (final URL wsdlURL : wsdlURLs) {
            try {
                // check if this conversion is necessary.
                final URI wsdlURI = new URI(wsdlURL.toString());
                final Definition wsdlDefinition = factory.newWSDLReader().readWSDL(new CatalogWSDLLocator(wsdlURI.toString()));

                List<SchemaType> schemaTypes = getExtendingTypes(SchemaUtils.getSchemas(wsdlDefinition), sourceUrl, elName, encoding);
                for (SchemaType schemaType : schemaTypes) {
                    returnedTypeList.add(schemaType.getName().getLocalPart());
                }
            } catch (final URISyntaxException e) {
                throw new WsdlDatasenseException(e);
            } catch (final WSDLException e) {
                throw new WsdlDatasenseException(e);
            } catch (TransformerException e) {
                throw new WsdlDatasenseException(e);
            }
        }

        return returnedTypeList;
    }

    public static SchemaType getExtendingType(List<String> listOfschemas, SchemaType propertyType, final String extendingType) {
        try {
            SchemaTypeSystem schemaTypeLoader = XmlSchemaUtils.getSchemaTypeSystem(listOfschemas, null);
            SchemaType[] listOfElements = schemaTypeLoader.globalTypes();

            if (!hasSimpleContentOnly(propertyType)) {
                return Iterables.find(Arrays.asList(listOfElements), new Predicate<SchemaType>() {

                    @Override
                    public boolean apply(SchemaType input) {
                        return input.getName().getLocalPart().equals(extendingType);
                    }
                });

            }
            return null;
        } catch (XmlException e) {
            throw new MetaDataGenerationException(e);
        }
    }

    public static String makeReadable(final String original) {
        // Change underscore separator for spaces
        String temporary = original.replace('_', ' ');

        // Separate camelized words
        String[] strings = StringUtils.splitByCharacterTypeCamelCase(temporary);

        // Filter out spaces and join it all back into a string
        return WordUtils.capitalizeFully(join(Iterables.filter(Lists.newArrayList(strings), NON_EMPTY_STRING), " "));
    }

    public static boolean hasSimpleContentOnly(SchemaType type) {

        return hasSimpleContent(type) && ArrayUtils.isEmpty(type.getAttributeProperties()) && ArrayUtils.isEmpty(type.getElementProperties());
    }

    public static boolean hasSimpleContent(SchemaType type) {

        return type.isSimpleType() || type.getContentType() == SchemaType.SIMPLE_CONTENT || type.getContentType() == SchemaType.MIXED_CONTENT
                || type.getComponentType() == SchemaType.EMPTY_CONTENT;
    }

}