package org.mule.modules.wsdl.runtime;

import org.apache.commons.lang.StringUtils;
import org.jetbrains.annotations.NotNull;

import static org.mule.modules.wsdl.datasense.SchemaUtils.DEFINITION_SEPARATOR;

/**
 * Models a SOAP service endpoint
 */
public class ServiceDefinition {

    public static final int DEFINITION_LENGTH = 3;
    /**
     * The namespace of the WSDL file.
     */
    private final String namespace;

    private final String serviceName;

    private final String portName;

    /**
     * The base path for the WSDL. Appending (WSDLname)/?wsdl to this should return the WSDL.
     */
    private final String baseEndpoint;

    /**
     * The QName prefix to be used when creating the header.
     */

    private final String headerPrefix;

    private ServiceDefinition(@NotNull final String namespace, @NotNull final String serviceName, @NotNull final String portName, @NotNull final String baseEndpoint,
            @NotNull final String headerPrefix) {
        this.namespace = namespace;
        this.serviceName = serviceName;
        this.portName = portName;
        this.baseEndpoint = baseEndpoint;
        this.headerPrefix = headerPrefix;
    }

    public static ServiceDefinition create(@NotNull final String namespace, @NotNull final String serviceName, @NotNull final String portName, @NotNull final String baseEndpoint,
            @NotNull final String headerPrefix) {
        return new ServiceDefinition(namespace, serviceName, portName, baseEndpoint, headerPrefix);
    }

    public static ServiceDefinition create(@NotNull String definition, @NotNull final String baseEndpoint, @NotNull final String headerPrefix) throws SoapCallException {

        final String definitionDetails[] = StringUtils.split(definition, DEFINITION_SEPARATOR);
        if (!definition.contains(DEFINITION_SEPARATOR) || definitionDetails.length < DEFINITION_LENGTH) {
            throw SoapCallException.createMetadataInvocationException(definition);
        }
        final String serviceName = definitionDetails[0];
        final String portName = definitionDetails[1];
        final String nameSpace = definitionDetails[2];
        if (StringUtils.isEmpty(serviceName) || StringUtils.isEmpty(portName) || StringUtils.isEmpty(nameSpace) ) {
            throw SoapCallException.createMetadataInvocationException(definition);
        }

        return new ServiceDefinition(nameSpace, serviceName, portName, baseEndpoint, headerPrefix);
    }

    @NotNull
    public String getNamespace() {
        return namespace;
    }

    @NotNull
    public String getServiceName() {
        return serviceName;
    }

    @NotNull
    public String getPortName() {
        return portName;
    }

    @NotNull
    public String getBaseEndpoint() {
        return baseEndpoint;
    }

    @NotNull
    public String getHeaderPrefix() {
        return headerPrefix;
    }

    @Override
    public String toString() {
        return "ServiceDefinition{" + "namespace='" + namespace + '\'' + ", serviceName='" + serviceName + '\'' + ", portName='" + portName + '\'' + ", baseEndpoint='"
                + baseEndpoint + '\'' + ", headerPrefix='" + headerPrefix + '\'' + '}';
    }

}
