/**
 * (c) 2003-2015 MuleSoft, Inc. This software is protected under international
 * copyright law. All use of this software is subject to MuleSoft's Master
 * Subscription Agreement (or other Terms of Service) separately entered
 * into between you and MuleSoft. If such an agreement is not in
 * place, you may not use the software.
 */
package org.mule.modules.wsdl.runtime;

import java.util.Collections;
import java.util.Map;

import javax.xml.namespace.QName;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.soap.MessageFactory;
import javax.xml.soap.SOAPBody;
import javax.xml.soap.SOAPBodyElement;
import javax.xml.soap.SOAPElement;
import javax.xml.soap.SOAPEnvelope;
import javax.xml.soap.SOAPException;
import javax.xml.soap.SOAPHeader;
import javax.xml.soap.SOAPMessage;
import javax.xml.soap.SOAPPart;
import javax.xml.stream.XMLStreamException;
import javax.xml.stream.XMLStreamReader;
import javax.xml.ws.BindingProvider;
import javax.xml.ws.Dispatch;
import javax.xml.ws.Service;
import javax.xml.ws.soap.SOAPBinding;

import org.apache.cxf.message.Message;
import org.jetbrains.annotations.NotNull;
import org.jetbrains.annotations.Nullable;
import org.mule.modules.wsdl.runtime.header.HeaderBuilder;
import org.mule.modules.wsdl.runtime.header.SoapHeaderException;
import org.mule.modules.wsdl.runtime.request.DocumentBuilder;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.w3c.dom.Document;

/**
 * Helper class that invokes SOAP web services.
 *
 * @author martin.paoloni@mulesoft.com
 */
public class SoapClient {

    private static final String XMLSOAP_ORG_SOAP_ENCODING_NAMESPACE = "http://schemas.xmlsoap.org/soap/encoding/";
    private static final Logger logger = LoggerFactory.getLogger(SoapClient.class);
    private static final String HEADER_PREFIX = "headerPrefix";
    private static final String BODY_PREFIX = "bodyPrefix";

    private final ServiceDefinition serviceDefinition;

    /**
     * The {@link HeaderBuilder} implementation. Optional.
     */
    private HeaderBuilder soapHeaderBuilder;

    /**
     * The parameters to be appended to the header.
     */
    private Map<String, String> headerParams;

    private Integer connectionTimeout;

    private Integer readTimeout;

    private SoapClient(@NotNull final ServiceDefinition service) {
        serviceDefinition = service;
    }

    @NotNull
    public static SoapClient create(@NotNull final ServiceDefinition service) {
        return new SoapClient(service);
    }

    /**
     * Invokes a web serviceDefinition.
     *
     * @param payload        A {link {@link XMLStreamReader reader} with the payload message to send.
     * @param callDefinition SOAP operation invocation definition
     * @return A {link {@link XMLStreamReader reader} with the output message received.
     * @throws SoapCallException Unexpected error during serviceDefinition call
     */
    @Nullable
    public XMLStreamReader invoke(@NotNull final CallDefinition callDefinition, @Nullable final XMLStreamReader payload) throws SoapCallException {
        return invoke(callDefinition, payload, new DocumentBuilder(null, null));
    }

    /**
     * Invokes a web serviceDefinition.
     *
     * @param payload        A {link {@link XMLStreamReader reader} with the payload message to send.
     * @param callDefinition SOAP operation invocation definition
     * @param docBuilder     class used to transform the reader into a Document and, if necessary, enrich it with an xsi:type attribute
     * @return A {link {@link XMLStreamReader reader} with the output message received.
     * @throws SoapCallException Unexpected error during serviceDefinition call
     */
    @Nullable
    public XMLStreamReader invoke(@NotNull final CallDefinition callDefinition, @Nullable final XMLStreamReader payload, DocumentBuilder docBuilder) throws SoapCallException {

        logger.debug("Service Definition for Invocation: {}", serviceDefinition);
        logger.debug("SOAP call to endpoint: {}", callDefinition);

        XMLStreamReader result= null;
        try {
            // Create dispatcher client ...
            final String operationName = callDefinition.getOperationName();
            final String endpointPath = callDefinition.getEndpointPath();
            final Dispatch<SOAPMessage> dispatch = buildMessageDispatch(endpointPath, operationName);

            // Build request ...
            final SOAPMessage soapRequest = buildSoapRequest(payload, operationName, dispatch, docBuilder);

            // Invoke the endpoint synchronously
            final SOAPMessage soapResponse = dispatch.invoke(soapRequest);

            logger.debug("Client call successful.");

            // If the serviceDefinition responds a 202 status, the payload will be null.
            // Therefore soapResponse is null. See org.apache.cxf.endpoint.ClientImpl,line 609.
            if (soapResponse != null && !soapResponse.getSOAPBody().hasFault())  {
                final SOAPBodyElement sourceContent = (SOAPBodyElement) soapResponse.getSOAPBody().getChildElements().next();

                // Add the namespaces from the response Envelope tag to the result
                // tag - so it can be parsed successfully by DataMapper
                final SOAPPart soapPart = soapRequest.getSOAPPart();
                final SOAPEnvelope soapEnvelope = soapPart.getEnvelope();

                result = XmlConverterUtils.soapResponseToXmlStream(soapResponse, sourceContent, soapEnvelope);
            }
        } catch (final Exception e) {
            logger.warn("Error during web serviceDefinition invocation", e);
            throw SoapCallException.createCallException(e);
        }

        return result;
    }

    @NotNull
    public Dispatch<SOAPMessage> buildMessageDispatch(@NotNull final String endpointPath, @NotNull final String operationName) {

        // QNames for serviceDefinition as defined in wsdl.
        final String namespace = serviceDefinition.getNamespace();
        final String operation = serviceDefinition.getServiceName();
        final QName serviceQName = new QName(namespace, operation);

        // QName for Port As defined in wsdl.
        final String portName = serviceDefinition.getPortName();
        final QName portQName = new QName(namespace, portName);

        // Endpoint Address
        final String baseEndpoint = serviceDefinition.getBaseEndpoint();
        final String endpointAddress = baseEndpoint + endpointPath;

        // Create a dynamic Service instance
        final Service service = Service.create(serviceQName);

        // Add a port to the Service
        service.addPort(portQName, SOAPBinding.SOAP11HTTP_BINDING, endpointAddress);

        // Create a dispatch instance
        final Dispatch<SOAPMessage> dispatch = service.createDispatch(portQName, SOAPMessage.class, Service.Mode.MESSAGE);

        // Optionally Configure RequestContext to send SOAPAction HTTP Header
        final Map<String, Object> rc = dispatch.getRequestContext();
        rc.put(BindingProvider.SOAPACTION_USE_PROPERTY, Boolean.TRUE);
        rc.put(BindingProvider.SOAPACTION_URI_PROPERTY, operationName);

        // Set connection and read timeout
        final Integer ctxTimeout = getConnectionTimeout();
        if (ctxTimeout != null) {
            rc.put(Message.CONNECTION_TIMEOUT, ctxTimeout);
            logger.debug("Setting timeout to {}", ctxTimeout);
        }

        final Integer rTimeout = getReadTimeout();
        if (rTimeout != null) {
            rc.put(Message.RECEIVE_TIMEOUT, rTimeout);
            logger.debug("Setting readTime to {}", rTimeout);
        }
        return dispatch;
    }

    /**
     * Build SOAP Request based on the provided parameters
     *
     * @param payload       Full payload to be sent without the operation
     * @param operationName Operation name to be invoked.
     * @param dispatch      Dispatch object used for the invocation.
     * @param docBuilder    class used to transform the reader into a Document and, if necessary, enrich it with an xsi:type attribute
     * @return SOAPMessage ready to be sent.
     * @throws SOAPException                when it cannot construct the SOAPMessage
     * @throws SoapHeaderException          when it cannot build the SOAP Headers
     * @throws XMLStreamException           when it cannot successfully create the XMLStreamReader
     * @throws ParserConfigurationException when it cannot properly modify the payload with the xsi:type attribute
     */
    @NotNull
    public SOAPMessage buildSoapRequest(@Nullable final XMLStreamReader payload, @NotNull final String operationName, @NotNull final Dispatch<SOAPMessage> dispatch,
            DocumentBuilder docBuilder) throws SOAPException, SoapHeaderException, XMLStreamException, ParserConfigurationException {

        // Obtain a pre-configured SAAJ MessageFactory
        final SOAPBinding binding = (SOAPBinding) dispatch.getBinding();
        final MessageFactory msgFactory = binding.getMessageFactory();

        // Create SOAPMessage Request
        final SOAPMessage requestMessage = msgFactory.createMessage();

        final SOAPPart part = requestMessage.getSOAPPart();

        // Gets the elements SOAPEnvelope, header and body.
        final SOAPEnvelope env = part.getEnvelope();
        env.setEncodingStyle(XMLSOAP_ORG_SOAP_ENCODING_NAMESPACE);

        final String namespace = serviceDefinition.getNamespace();
        env.addNamespaceDeclaration(BODY_PREFIX, namespace + operationName + "/");
        env.addNamespaceDeclaration(HEADER_PREFIX, namespace);

        // Request Header
        final SOAPHeader header = env.getHeader();
        if (soapHeaderBuilder != null) {
            soapHeaderBuilder.build(header, serviceDefinition);
        }

        final Map<String, String> params = getHeaderParams();
        final String headerPrefix = serviceDefinition.getHeaderPrefix();
        for (final String key : params.keySet()) {
            final QName qname = new QName(namespace, key, headerPrefix);
            final SOAPElement soapElement = header.addChildElement(qname);
            final String headerName = params.get(key);
            soapElement.addTextNode(headerName);
        }

        XMLStreamReader callsPayload = payload;
        // Compose the soap:Body payload
        if (callsPayload == null) {
            String soapMethodsCallNamespace = namespace.endsWith("/") ? namespace.substring(0, namespace.length() - 1) : namespace;
            callsPayload = XmlConverterUtils.computeCallsPayloadForMethodWithNoParameter(operationName, soapMethodsCallNamespace);
        }
        /*
         * We created an enhancement request(DEVKIT-2182) in order for the xsi:type attribute to be added at dataSense. Until then, we will use the ComplexDocumentBuilder to add it
         * at runtime
         */
        final Document document = docBuilder.createDocument(callsPayload);
        final SOAPBody body = env.getBody();
        body.addDocument(document);
        requestMessage.saveChanges();

        return requestMessage;
    }

    /**
     * Getter method for connection timeout.
     *
     * @return the connectionTimeout
     */
    public Integer getConnectionTimeout() {
        return connectionTimeout;
    }

    /**
     * Setter method for connectionTimeout.
     *
     * @param connectionTimeout the connectionTimeout to set
     */
    public void setConnectionTimeout(final int connectionTimeout) {
        this.connectionTimeout = connectionTimeout;
    }

    /**
     * Getter method for readTimeout
     *
     * @return the readTimeout
     */
    public Integer getReadTimeout() {
        return readTimeout;
    }

    /**
     * Setter method for readTimeout.
     *
     * @param readTimeout the readTimeout to set
     */
    public void setReadTimeout(final Integer readTimeout) {
        this.readTimeout = readTimeout;
    }

    @NotNull
    public Map<String, String> getHeaderParams() {
        return headerParams != null ? headerParams : Collections.<String, String>emptyMap();
    }

    /**
     * Sets the parameters to be appended to the header.
     *
     * @param headerParams the header parameters to set.
     */
    public void setHeaderParams(@NotNull final Map<String, String> headerParams) {
        this.headerParams = headerParams;
    }

    /**
     * Sets the {@link HeaderBuilder} implementation.
     *
     * @param soapHeaderBuilder the {@link HeaderBuilder} implementation.
     */
    public void setSoapHeaderBuilder(final HeaderBuilder soapHeaderBuilder) {
        this.soapHeaderBuilder = soapHeaderBuilder;
    }
}
