%dw 2.0
import firstWith from dw::core::Arrays
import fail from dw::Runtime
import * from jsonschema!agent_network

ns xsi http://www.w3.org/2001/XMLSchema-instance
ns tls http://www.mulesoft.org/schema/mule/tls
ns http http://www.mulesoft.org/schema/mule/http
ns a2a http://www.mulesoft.org/schema/mule/a2a
ns ms_einstein_ai http://www.mulesoft.org/schema/mule/ms-einstein-ai
ns mcp http://www.mulesoft.org/schema/mule/mcp
ns agents_broker http://www.mulesoft.org/schema/mule/agents-broker
ns ee http://www.mulesoft.org/schema/mule/ee/core
ns oauth http://www.mulesoft.org/schema/mule/oauth


type GAV = {
  groupId: String,
  assetId: String,
  version: String,
}


type Exchange = {
  main: String,
  name: String,
  groupId: String,
  assetId: String,
  version: String,
  dependencies?: Array<GAV>,
  metadata?: {}      
}

var MCP_CLIENT_CONFIG_SUFIX = "mcp-client-config"
var A2A_CLIENT_CONFIG_SUFIX = "a2a-client-config"

fun resolveReference(namespace: String, name: String, exchange: Exchange): GAV | Null = do {
    exchange.dependencies firstWith ((dep) -> dep.assetId == name and dep.groupId == namespace)
  }

fun createExternalMCPClientConfig(agents: { _?: Broker }, mcpResolver: (namespace: String, name: String) -> { transport: MCPTransport } | Null, exchange: Exchange, lowerEnv: Boolean) =  
  agents pluck ((agent, agentName, index) -> do {
          { 
            (
              agent.spec.tools map ((item, index) -> do {                        
                      var connection = item.mcp.connection.ref.name onNull "\${connections.$(item.mcp.ref.name).default}"                                     
                      var mcpRef = item.mcp.ref                    
                      var namespace = resolveReferenceNamespace(mcpRef, exchange)  
                      var toolTransport = mcpResolver(namespace, item.mcp.ref.name)
                      ---
                      {
                        mcp#"client-config" @(name: "$(agentName)-$(item.mcp.ref.name)-$(MCP_CLIENT_CONFIG_SUFIX)", clientName: item.mcp.ref.name, clientVersion: "1.0.0"): 
                          if(toolTransport.transport.kind == "sse") 
                            {
                              mcp#"sse-client-connection" @(serverUrl: "\${egressgw.url}/$(namespace)/$(item.mcp.ref.name)/$(connection)", sseEndpointPath: toolTransport.transport.ssePath, requestTimeout: agent.spec.taskTimeoutSecs onNull 60): 
                                if(lowerEnv)
                                  { 
                                    tls#context: {
                                        tls#"trust-store" @(insecure: "true"): {}
                                      } 
                                  }
                                else {}
                            }                        
                          else 
                            {
                              mcp#"streamable-http-client-connection" @(serverUrl: "\${egressgw.url}/$(namespace)/$(item.mcp.ref.name)/$(connection)", (mcpEndpointPath: toolTransport.transport.streamableHttp.path) if(toolTransport.transport.streamableHttp.path?), requestTimeout: agent.spec.taskTimeoutSecs onNull 60): 
                                if(lowerEnv)
                                  { 
                                    tls#context: {
                                        tls#"trust-store" @(insecure: "true"): {}
                                      } 
                                  }
                                else {}
                            }    
                      }
                    })
            )
          }
        })

fun createLLMConnection(llm: BrokerLLM, exchange: Exchange, lowerEnv: Boolean) = do {
    var connection = llm.connection.ref.name onNull "\${connections.$(llm.ref.name).default}"                                         
    var namespace = resolveReferenceNamespace(llm.ref, exchange)                      
    ---
    {   
      agents_broker#"openai-connection" @(modelName: llm.configuration.model, apiKey: "-", url: "\${egressgw.url}/$(namespace)/$(llm.ref.name)/$(connection)"):
        if(lowerEnv)
          {
            tls#context: {
                tls#"trust-store" @(insecure: "true"): {}
              }
          }
        else {}
    }        
  }


fun resolveReferenceNamespace(agentRef: { name: String, namespace?: String }, exchange: Exchange) = 
  agentRef.namespace onNull (() -> exchange.dependencies firstWith ((dep) -> dep.assetId == agentRef.name) then $.groupId onNull (() -> exchange.groupId))

fun createExternalAgentClientConfig(agents: { _?: Broker }, exchange: Exchange, lowerEnv: Boolean) =    
  agents pluck ((agent, agentName, index) -> do {
          { 
            (
              agent.spec.links 
                map ((item, index) -> do {   
                      var connection = item.agent.connection.ref.name onNull "\${connections.$(item.agent.ref.name).default}"                                     
                      var agentRef = item.agent.ref                    
                      var namespace = resolveReferenceNamespace(agentRef, exchange)                      
                      ---
                      {
                        a2a#"client-config" @(name: "$(agentName)-$(agentRef.name)-$(A2A_CLIENT_CONFIG_SUFIX)"): {                            
                            a2a#"client-connection" @(serverUrl: "\${egressgw.url}/$(namespace)/$(agentRef.name)/$(connection)", requestTimeout: agent.spec.taskTimeoutSecs onNull 60):                             
                          //TODO we need to set the clientId and clientSecret of this conductor
                          //a2a#"default-headers": {                        
                          //a2a#"default-header" @(key: "clientId", value: value.authentication.clientId): {},
                          //a2a#"default-header" @(key: "clientSecret", value: value.authentication.clientSecret): {}                                
                          //}
                              if(lowerEnv)
                                { 
                                  tls#context: {
                                      tls#"trust-store" @(insecure: "true"): {}
                                    } 
                                } 
                              else {}
                          }
                      }
                    })
            )
          }
        })
                  
           

fun createSecurityScheme(securitySchemeName: String, securityScheme: SecurityScheme) =
  //TODO add support for other security schemes
  securityScheme  match {    
    case oauth is OAuth2SecurityScheme -> 
      {
        a2a#"oauth2-security-scheme" @(securitySchemeName: securitySchemeName): {
            a2a#flows: {
                (a2a#"client-credentials" @(tokenUrl: securityScheme.flows.clientCredentials.tokenUrl): {
                    a2a#"oauth2-scopes": {
                        (oauth.flows.clientCredentials.scopes pluck ((scope, key, index) -> {
                                  a2a#"oauth2-scope" @(name: scope): {
                                      a2a#description: scope as CData
                                    }
                                }
                              )
                          )
                      }
                  }) if (securityScheme.flows.clientCredentials?),
              }
          }
      }      
    case apikey is APIKeySecurityScheme -> 
      {
        a2a#"api-key-security-scheme" @(name: apikey.name, in: upper(apikey.in), securitySchemeName: securitySchemeName): {
            a2a#description: (apikey.description default "") as CData,
          }
      }
    case securityScheme is HTTPAuthSecurityScheme -> 
      {
        a2a#"http-security-scheme" @(securitySchemeName: securitySchemeName, scheme: securityScheme.scheme, (bearerFormat: securityScheme.bearerFormat) if(securityScheme.bearerFormat?)): {
            a2a#description: (securityScheme.description default "") as CData,
          }
      }
    case securityScheme is OpenIdConnectSecurityScheme -> 
      {
        a2a#"open-id-connect-security-scheme" @(securitySchemeName: securitySchemeName, openIdConnectUrl: securityScheme.openIdConnectUrl): {
            a2a#description: (securityScheme.description default "") as CData,
          }
      }
    else -> fail("Unsupported security scheme type: " ++ (typeOf(securityScheme) as String))
  }
  
fun getConductorConfigName(agent: Broker) = agent.card.name replace " " with ("_")

fun createConductorConfig(brokers: { _?: Broker }, exchange: Exchange, lowerEnv: Boolean) =
  brokers pluck ((broker, agentName, index) -> {
          agents_broker#"config" @(
            name: getConductorConfigName(broker), 
            (maxNumberOfLoops: broker.spec.maxNumberOfLoops) if (broker.spec.maxNumberOfLoops != null), 
            (maxConsecutiveErrors: broker.spec.maxConsecutiveErrors) if (broker.spec.maxConsecutiveErrors != null)): createLLMConnection(broker.spec.llm, exchange, lowerEnv)
        })
  
fun createAgentListenerConfig(brokers: { _?: Broker }) =
  brokers pluck ((agent, agentName, index) -> {
          a2a#"server-config" @(name: "$(agentName)-a2a-listener-config"): {
              a2a#connection @(listenerConfig: "http-listener-config", agentPath: "/$(agentName)"): null,
              a2a#card @(name: agent.card.name, url: "http://localhost:8081/$(agentName)", version: "1.0.0", protocolVersion: agent.card.protocolVersion): {
                  a2a#description: agent.card.description as CData,                 
                  a2a#skills: {
                      (agent.card.skills map ((skill) -> {
                                a2a#"agent-skill" @(id: skill.id, name: skill.name): {
                                    a2a#description: skill.description as CData,
                                    (a2a#tags: {
                                        (skill.tags map ((tag) -> {
                                                  a2a#tag @(value: tag): {}
                                                }))
                                      }) if (!isEmpty(skill.tags)),
                                    (a2a#examples: {
                                        (skill.examples map ((example) -> {
                                                  a2a#example @(value: example): {}
                                                }))
                                      }) if (!isEmpty(skill.examples)),
                                    (a2a#"input-modes": {
                                        (skill.inputModes map ((inputMode) -> {
                                                  a2a#"input-mode" @(value: inputMode): {}
                                                }))
                                      }) if (!isEmpty(skill.inputModes)),
                                    (a2a#"output-modes": {
                                        (skill.outputModes map ((outputMode) -> {
                                                  a2a#"output-mode" @(value: outputMode): {}
                                                }))
                                      }) if (!isEmpty(skill.outputModes))
                                  }
                              }))
                    },
                  (if (agent.card.capabilities.streaming default false) fail("Agent brokers not yet support streaming")
                   else if (agent.card.capabilities.stateTransitionHistory default false) fail("Agent brokers not yet support state transition history")
                   else a2a#capabilities @(streaming: false, 
                                          stateTransitionHistory: false,
                                          pushNotifications: agent.card.capabilities.pushNotifications default true): {
                                            (a2a#extensions: {
                                              (agent.card.capabilities.extensions map ((extension) -> {
                                                a2a#extension @(uri: extension.uri, required: extension.required default false): {
                                                  a2a#description: (extension.description default "") as CData,
                                                  a2a#params: {
                                                    (extension.params pluck ((value, key, index) -> {
                                                      a2a#param @(key: key, value: value): null
                                                    }))
                                                  }
                                                }
                                              }))
                                            }) if (!isEmpty(agent.card.capabilities.extensions))
                                          }
                  ),                  
                  a2a#provider @(organization: agent.card.provider.organization, url: agent.provider.url): null,
                  a2a#"default-input-modes": {
                      (agent.card.defaultInputModes map ((inputMode) -> {
                                a2a#"default-input-mode" @(value: inputMode): {}
                              }))
                    },
                  a2a#"default-output-modes": {
                      (agent.card.defaultOutputModes map ((outputMode) -> {
                                a2a#"default-output-mode" @(value: outputMode): {}
                              }))
                    },
                  (a2a#security: {
                      (agent.card.security map ((schemeDefinition, index) -> do {
                                var schemeName = keysOf(schemeDefinition)[0]
                                var scheme = schemeDefinition[schemeName]
                                ---
                                a2a#"security-requirement" @(securitySchemeName: keysOf(schemeDefinition)[0]): {
                                      a2a#scopes: {
                                          (scheme map ((security) -> {
                                                    a2a#scope @(value: security): {}
                                                  }))
                                        }
                                    }
                              }))
                    }) if (agent.card.security?),
                  (a2a#"security-schemes": {
                      (agent.card.securitySchemes pluck ((securityScheme, securitySchemeName, index) ->
                              createSecurityScheme(securitySchemeName, securityScheme)
                            ))
                    }) if (agent.card.securitySchemes?)
                }
            }
        })



fun createBrokersFlows(agents: { _?: Broker }) =
  agents pluck ((value, key, index) -> createBrokerFlow(key as String, value, agents))

fun createBrokerFlow(brokerName: String, agent: Broker, allAgents: { _?: Broker }) =
  {
    flow @(name: "$(brokerName)-flow"): {
        a2a#"task-listener" @("config-ref": "$(brokerName)-a2a-listener-config"): null,
            agents_broker#"agent-loop" @(
              taskId: "#[attributes.taskId]", 
              contextId: "#[attributes.contextId]", 
              apiInstanceId: "#[attributes.headers.'x-anypoint-api-instance-id']",
              "config-ref": getConductorConfigName(agent)): {
                agents_broker#prompt: "#[payload.message.parts.text joinBy \"\\n\" onNull \"\"]" as CData,               
                (agents_broker#instructions: (agent.spec.instructions joinBy "\n") as CData) if (!isEmpty(agent.spec.instructions)),
                (agents_broker#"mcp-servers": { (
                        agent.spec.tools flatMap ((tool) ->                              
                              [ 
                                {
                                  agents_broker#"mcp-server" @(mcpClientConfigRef: "$(brokerName)-$(tool.mcp.ref.name)-$(MCP_CLIENT_CONFIG_SUFIX)"): {
                                      (agents_broker#"allowed-tools": {
                                          (tool.allowed map ((tool) -> {
                                                    agents_broker#"allowed-tool" @(value: tool): null
                                                  }))
                                        }) if (!isEmpty(tool.allowed))
                                    }
                                }
                              ]                              
                            )
          )}),
                (agents_broker#"a2a-clients": {
                    (agent.spec.links map ((linkDef) -> {                              
                              agents_broker#"a2a-client" @(a2aClientConfigRef: "$(brokerName)-$(linkDef.agent.ref.name)-$(A2A_CLIENT_CONFIG_SUFIX)"): {}                              
                            }))
                  }) if (!isEmpty(agent.spec.links))
              }
      }
  }


fun createHttpListenerConfig(brokers: { _?: Broker }) =
  {
    http#"listener-config" @(name: "http-listener-config"): {
        http#"listener-connection" @(
          host: "0.0.0.0", 
          port: "\${http.port}", 
          readTimeout: (max(brokers pluck ((value, key, index) -> value.spec.taskTimeoutSecs onNull 60)) onNull 60) * 1000
          ): {}
      }
  }

fun createConfigurationElement() =
  {
    "configuration-properties" @(file: "config.yaml"): {}
  }
