%dw 2.0
import fail from dw::Runtime
import * from jsonschema!agent_network

ns xsi http://www.w3.org/2001/XMLSchema-instance
ns tls http://www.mulesoft.org/schema/mule/tls
ns http http://www.mulesoft.org/schema/mule/http
ns a2a http://www.mulesoft.org/schema/mule/a2a
ns ms_einstein_ai http://www.mulesoft.org/schema/mule/ms-einstein-ai
ns mcp http://www.mulesoft.org/schema/mule/mcp
ns agents_broker http://www.mulesoft.org/schema/mule/agents-broker
ns ee http://www.mulesoft.org/schema/mule/ee/core
ns oauth http://www.mulesoft.org/schema/mule/oauth

type Exchange = {
  main: String,
  name: String,
  groupId: String,
  assetId: String,
  version: String,
  dependencies?: Array<{
        groupId: String,
        assetId: String,
        version: String,
      }>,
  metadata?: {}      
}

fun createExternalMCPClientConfig(agents: { _?: Broker }, mcps: { _?: { transport: MCPTransport } }, exchange: Exchange, lowerEnv: Boolean) =  
  agents pluck ((agent, agentName, index) -> do {
          { 
            (
              agent.spec.tools map ((item, index) -> do {                        
                      var connection = item.connectionRef.name onNull "\${connections.$(item.mcp.ref.name).default}"                                     
                      var mcpRef = item.mcp.ref                    
                      var namespace = resolveReferenceNamespace(mcpRef, exchange)  
                      var toolTransport = mcps[namespace ++ "/" ++ item.mcp.ref.name]
                      ---
                      {
                        mcp#"client-config" @(name: "$(agentName)-$(item.mcp.ref.name)-mcp-client-config", clientName: item.mcp.ref.name, clientVersion: "1.0.0"): 
                          if(toolTransport.transport.kind == "sse") 
                            {
                              mcp#"sse-client-connection" @(serverUrl: "\${egressgw.url}/$(namespace)/$(item.mcp.ref.name)/$(connection)", sseEndpointPath: toolTransport.transport.sse.ssePath): 
                                if(lowerEnv)
                                  { 
                                    tls#context: {
                                        tls#"trust-store" @(insecure: "true"): {}
                                      } 
                                  }
                                else {}
                            }                        
                          else 
                            {
                              mcp#"streamable-http-client-connection" @(serverUrl: "\${egressgw.url}/$(namespace)/$(item.mcp.ref.name)/$(connection)", (mcpEndpointPath: toolTransport.transport.streamableHttp.path) if(toolTransport.transport.streamableHttp.path?)): 
                                if(lowerEnv)
                                  { 
                                    tls#context: {
                                        tls#"trust-store" @(insecure: "true"): {}
                                      } 
                                  }
                                else {}
                            }    
                      }
                    })
            )
          }
        })

fun createLLMConnection(llm: BrokerLLM, exchange: Exchange) = do {
    var connection = llm.connection.ref.name onNull "\${connections.$(llm.ref.name).default}"                                         
    var namespace = resolveReferenceNamespace(llm.ref, exchange)                      
    ---
    {   
      agents_broker#"openai-connection" @(modelName: llm.configuration.model, apiKey: "-", url: "\${egressgw.url}/$(namespace)/$(llm.ref.name)/$(connection)"): {}    
    }        
  }


fun resolveReferenceNamespace(agentRef: RegistryRef, exchange: Exchange) = 
  agentRef.namespace onNull (() -> exchange.dependencies find ((dep) -> dep.assetId == agentRef.name) then $.groupId onNull (() -> exchange.groupId))

fun createExternalAgentClientConfig(agents: { _?: Broker }, exchange: Exchange, lowerEnv: Boolean) =    
  agents pluck ((agent, agentName, index) -> do {
          { 
            (
              agent.spec.links 
                map ((item, index) -> do {   
                      var connection = item.connectionRef.name onNull "\${connections.$(item.agent.ref.name).default}"                                     
                      var agentRef = item.agent.ref                    
                      var namespace = resolveReferenceNamespace(agentRef, exchange)                      
                      ---
                      {
                        a2a#"client-config" @(name: "$(agentName)-$(agentRef.name)-a2a-client-config"): {                            
                            a2a#"client-connection" @(serverUrl: "\${egressgw.url}/$(namespace)/$(agentRef.name)/$(connection)"):                             
                          //TODO we need to set the clientId and clientSecret of this conductor
                          //a2a#"default-headers": {                        
                          //a2a#"default-header" @(key: "clientId", value: value.authentication.clientId): {},
                          //a2a#"default-header" @(key: "clientSecret", value: value.authentication.clientSecret): {}                                
                          //}
                              if(lowerEnv)
                                { 
                                  tls#context: {
                                      tls#"trust-store" @(insecure: "true"): {}
                                    } 
                                } 
                              else {}
                          }
                      }
                    })
            )
          }
        })
                  
           

fun createSecurityScheme(securitySchemeName: String, securityScheme: SecurityScheme) =
  //TODO add support for other security schemes
  securityScheme  match {    
    case oauth is OAuth2SecurityScheme -> 
      {
        a2a#"oauth2-security-scheme" @(securitySchemeName: securitySchemeName): {
            a2a#flows: {
                (a2a#"client-credentials" @(tokenUrl: securityScheme.flows.clientCredentials.tokenUrl): {
                    a2a#"oauth2-scopes": {
                        (oauth.flows.clientCredentials.scopes pluck ((scope, key, index) -> {
                                  a2a#"oauth2-scope" @(name: scope): {
                                      a2a#description: scope as CData
                                    }
                                }
                              )
                          )
                      }
                  }) if (securityScheme.flows.clientCredentials?),
              }
          }
      }      
    case apikey is APIKeySecurityScheme -> 
      {
        a2a#"api-key-security-scheme" @(name: apikey.name, in: upper(apikey.in), securitySchemeName: securitySchemeName): {
            a2a#description: (apikey.description default "") as CData,
          }
      }
    case securityScheme is HTTPAuthSecurityScheme -> 
      {
        a2a#"http-security-scheme" @(securitySchemeName: securitySchemeName, scheme: securityScheme.scheme, (bearerFormat: securityScheme.bearerFormat) if(securityScheme.bearerFormat?)): {
            a2a#description: (securityScheme.description default "") as CData,
          }
      }
    case securityScheme is OpenIdConnectSecurityScheme -> 
      {
        a2a#"open-id-connect-security-scheme" @(securitySchemeName: securitySchemeName, openIdConnectUrl: securityScheme.openIdConnectUrl): {
            a2a#description: (securityScheme.description default "") as CData,
          }
      }
    else -> fail("Unsupported security scheme type: " ++ (typeOf(securityScheme) as String))
  }
  
fun getConductorConfigName(agent: Broker) = agent.card.name replace " " with ("_")

fun createConductorConfig(brokers: { _?: Broker }, exchange: Exchange) =
  brokers pluck ((broker, agentName, index) -> {
          agents_broker#"config" @(
            name: getConductorConfigName(broker), 
            (maxNumberOfLoops: broker.spec.maxNumberOfLoops) if (broker.spec.maxNumberOfLoops != null), 
            (maxConsecutiveErrors: broker.spec.maxConsecutiveErrors) if (broker.spec.maxConsecutiveErrors != null)): createLLMConnection(broker.spec.llm, exchange)
        })
  
fun createAgentListenerConfig(brokers: { _?: Broker }) =
  brokers pluck ((agent, agentName, index) -> {
          a2a#"server-config" @(name: "$(agentName)-a2a-listener-config"): {
              a2a#connection @(listenerConfig: "http-listener-config", agentPath: "/$(agentName)"): null,
              a2a#card @(name: agent.card.name, url: "http://localhost:8081/$(agentName)", version: "1.0.0"): {
                  a2a#description: agent.card.description as CData,
                  a2a#skills: {
                      (agent.card.skills map ((skill) -> {
                                a2a#"agent-skill" @(id: skill.id, name: skill.name): {
                                    a2a#description: skill.description as CData,
                                    (a2a#tags: {
                                        (skill.tags map ((tag) -> {
                                                  a2a#tag @(value: tag): {}
                                                }))
                                      }) if (!isEmpty(skill.tags)),
                                    (a2a#examples: {
                                        (skill.examples map ((example) -> {
                                                  a2a#example @(value: example): {}
                                                }))
                                      }) if (!isEmpty(skill.examples)),
                                    (a2a#"input-modes": {
                                        (skill.inputModes map ((inputMode) -> {
                                                  a2a#"input-mode" @(value: inputMode): {}
                                                }))
                                      }) if (!isEmpty(skill.inputModes)),
                                    (a2a#"output-modes": {
                                        (skill.outputModes map ((outputMode) -> {
                                                  a2a#"output-mode" @(value: outputMode): {}
                                                }))
                                      }) if (!isEmpty(skill.outputModes))
                                  }
                              }))
                    },
                  a2a#provider @(organization: agent.card.provider.organization, url: agent.provider.url): null,
                  a2a#"default-input-modes": {
                      (agent.card.defaultInputModes map ((inputMode) -> {
                                a2a#"default-input-mode" @(value: inputMode): {}
                              }))
                    },
                  a2a#"default-output-modes": {
                      (agent.card.defaultOutputModes map ((outputMode) -> {
                                a2a#"default-output-mode" @(value: outputMode): {}
                              }))
                    },
                  (a2a#security: {
                      (agent.card.security map ((schemeDefinition, index) -> do {
                                var schemeName = keysOf(schemeDefinition)[0]
                                var scheme = schemeDefinition[schemeName]
                                ---
                                a2a#"security-requirement" @(securitySchemeName: keysOf(schemeDefinition)[0]): {
                                      a2a#scopes: {
                                          (scheme map ((security) -> {
                                                    a2a#scope @(value: security): {}
                                                  }))
                                        }
                                    }
                              }))
                    }) if (agent.card.security?),
                  (a2a#"security-schemes": {
                      (agent.card.securitySchemes pluck ((securityScheme, securitySchemeName, index) ->
                              createSecurityScheme(securitySchemeName, securityScheme)
                            ))
                    }) if (agent.card.securitySchemes?)
                }
            }
        })



/**
* Describes the `createAgents` function purpose.
*
* === Parameters
*
* [%header, cols="1,1,3"]
* |===
* | Name | Type | Description
* | `agents` | { _?: Broker } | 
* |===
*
* === Example
*
* This example shows how the `createAgents` function behaves under different inputs.
*
* ==== Source
*
* [source,DataWeave,linenums]
* ----
* %dw 2.0
* output application/json
* ---
*
*
* ----
*
* ==== Output
*
* [source,Json,linenums]
* ----
*
* ----
*
*/
fun createBrokersFlows(agents: { _?: Broker }) =
  agents pluck ((value, key, index) -> createBrokerFlow(key as String, value, agents))

fun createBrokerFlow(brokerName: String, agent: Broker, allAgents: { _?: Broker }) =
  {
    flow @(name: "$(brokerName)-flow"): {
        a2a#"task-listener" @("config-ref": "$(brokerName)-a2a-listener-config"): null,
        try: {
            agents_broker#"agent-loop" @(
              taskId: "#[if (payload.message.taskId != null) payload.message.taskId else if (payload.message.referenceTaskIds != null and sizeOf(payload.message.referenceTaskIds) > 0) payload.message.referenceTaskIds[0] else null]", 
              contextId: "#[payload.message.contextId]", 
              "config-ref": getConductorConfigName(agent)): {
                agents_broker#prompt: "#[payload.message.parts.text joinBy \"\\n\" onNull \"\"]" as CData,               
                (agents_broker#instructions: (agent.spec.instructions joinBy "\n") as CData) if (!isEmpty(agent.spec.instructions)),
                (agents_broker#"mcp-servers": { (
                        agent.spec.tools flatMap ((tool) ->                              
                              [ 
                                {
                                  agents_broker#"mcp-server" @(mcpClientConfigRef: "$(brokerName)-$(tool.mcp.ref.name)-a2a-client-config"): {
                                      (agents_broker#"allowed-tools": {
                                          (tool.allowed map ((tool) -> {
                                                    agents_broker#"allowed-tool" @(value: tool): null
                                                  }))
                                        }) if (!isEmpty(tool.allowed))
                                    }
                                }
                              ]                              
                            )
          )}),
                (agents_broker#"a2a-clients": {
                    (agent.spec.links map ((linkDef) -> {                              
                              agents_broker#"a2a-client" @(a2aClientConfigRef: "$(brokerName)-$(linkDef.agent.ref.name)-a2a-client-config"): {}                              
                            }))
                  }) if (!isEmpty(agent.spec.links))
              },
            ee#transform: {
                ee#message: {
                    ee#"set-payload": "%dw 2.0\noutput application/json\n--- \n{\n    \"id\": payload.taskId,\n    \"contextId\": payload.contextId,\n    \"status\": {\n      \"state\": \n        if(payload.goalComplete) \"completed\" \n        else if (payload.inputRequired) \"input-required\" \n        else \"failed\",\n    },\n    \"artifacts\": [\n      {\n        \"artifactId\": uuid(),\n        \"parts\": [\n          {\n            \"kind\": \"text\",\n            \"text\": payload.response\n          }\n        ]\n      }\n    ],\n    \"kind\": \"task\"\n  }" as CData
                  }
              },
            "error-handler": {
                "on-error-continue" @(enableNotifications: "true", logException: "true"): {
                    ee#transform: {
                        ee#message: {
                            ee#"set-payload": "%dw 2.0\noutput application/json\n--- \n{\n    (\"id\": payload.id) if (payload.id?),\n    (\"contextId\": payload.contextId) if (payload.contextId?),\n    \"status\": {\n      \"state\": \"failed\",\n \"message\": {\n \"role\": \"agent\",\n \"messageId\": uuid(),\n \"parts\": [\n          {\n            \"kind\": \"text\",\n            \"text\": error.description\n          }\n        ]\n }  \n} ,\n    \"kind\": if (payload.id?) \"task\" else \"message\"\n  }" as CData
                          }
                      }
                  }
              }
          },
      }
  }


fun createHttpListenerConfig() =
  {
    http#"listener-config" @(name: "http-listener-config"): {
        http#"listener-connection" @(host: "0.0.0.0", port: "\${http.port}"): {}
      }
  }

fun createConfigurationElement() =
  {
    "configuration-properties" @(file: "config.yaml"): {}
  }
