package com.mulesoft.anypoint.agent.generator;

import org.mule.weave.v2.model.ServiceManager;
import org.mule.weave.v2.parser.ast.variables.NameIdentifier;
import org.mule.weave.v2.runtime.DataWeaveScript;
import org.mule.weave.v2.runtime.DataWeaveScriptingEngine;
import org.mule.weave.v2.runtime.ScriptingBindings;
import scala.Option;

import java.io.*;
import java.util.Arrays;
import java.util.Map;
import java.util.stream.Collectors;

public class MuleAppGenerator {

  public static final String MULE_ARTIFACT_JSON = "mule-artifact.json";
  public static final String LOG4J_CONFIG = "app-log4j.xml-template";
  public static final String MULE_AGENT_XML = "mule-agent.xml";

  public static final String APPLICATION_YAML = "application/yaml";
  public static final String APPLICATION_JSON = "application/json";
  public static final String APPLICATION_JAVA = "application/java";
  public static final String MVNW_CMD = "mvnw.cmd";
  public static final String MVNW = "mvnw";
  public static final String MAVEN_WRAPPER_PROPERTIES = "maven-wrapper.properties";
  public static final String GENERATE = "generate";
  public static final String PAYLOAD = "payload";

  private final DataWeaveScriptingEngine dataWeaveScriptingEngine = new DataWeaveScriptingEngine();

  public MuleAppGenerator() {
  }

  public void generateMuleApp(File mainFile, File exchangeJsonFile, Boolean lowerEnv, String modulesPath, File outputDirectory, Map<String, String> versions) {
    try {
      outputDirectory.mkdirs();
      final DataWeaveScript muleAppGenerator = dataWeaveScriptingEngine.compile(NameIdentifier.apply("MuleAppGenerator", Option.empty()));
      try (FileInputStream fabricFileStream = new FileInputStream(mainFile)) {
        final File srcMuleDir = new File(outputDirectory, "src" + File.separator + "main" + File.separator + "mule");
        srcMuleDir.mkdirs();
        final File outputFile = new File(srcMuleDir, MULE_AGENT_XML);
        final ScriptingBindings bindings = new ScriptingBindings()
            .addBinding(PAYLOAD, fabricFileStream, APPLICATION_YAML)
            .addBinding("exchange", exchangeJsonFile, APPLICATION_JSON)
            .addBinding("env", Map.of("lowerEnv", lowerEnv, "modulesPath", modulesPath), "application/java");
        muleAppGenerator.write(bindings, ServiceManager.apply(), Option.apply(outputFile));
      }

      final File resourcesMuleDir = new File(outputDirectory, "src" + File.separator + "main" + File.separator + "resources");
      resourcesMuleDir.mkdirs();
      final DataWeaveScript configYaml = dataWeaveScriptingEngine.compile(NameIdentifier.apply("MuleAppConfigGenerator", Option.empty()));

      try (FileInputStream exchangeJsonStream = new FileInputStream(exchangeJsonFile)) {
        final File outputFile = new File(resourcesMuleDir, "config.yaml");
        final ScriptingBindings bindings = new ScriptingBindings()
            .addBinding(PAYLOAD, exchangeJsonStream, APPLICATION_JSON)
            .addBinding("port", 8081, APPLICATION_JAVA)
            .addBinding("agentGW", "http://localhost:8081", APPLICATION_JAVA);
        configYaml.write(bindings, ServiceManager.apply(), Option.apply(outputFile));
      }

      writeResource(LOG4J_CONFIG, new File(resourcesMuleDir, "log4j2.xml"));

      final DataWeaveScript pomGenerator = dataWeaveScriptingEngine.compile(NameIdentifier.apply("MuleAppPomGenerator", Option.empty()));
      try (FileInputStream exchangeJsonStream = new FileInputStream(exchangeJsonFile)) {
        final File pom = new File(outputDirectory, "pom.xml");
        final ScriptingBindings bindings = new ScriptingBindings()
            .addBinding(PAYLOAD, exchangeJsonStream, APPLICATION_JSON)
            .addBinding("versions", versions, APPLICATION_JAVA);
        pomGenerator.write(bindings, ServiceManager.apply(), Option.apply(pom));
      }

      writeResource(MULE_ARTIFACT_JSON, new File(outputDirectory, MULE_ARTIFACT_JSON));

      try (InputStream mvnw = getClass().getClassLoader().getResourceAsStream(MVNW)) {
        assert mvnw != null;
        File file = new File(outputDirectory, MVNW);
        try (OutputStream mvnwOut = new FileOutputStream(file)) {
          mvnwOut.write(mvnw.readAllBytes());
          file.setExecutable(true);
        }
      }

      try (InputStream mavenWrapper = getClass().getClassLoader().getResourceAsStream(MAVEN_WRAPPER_PROPERTIES)) {
        assert mavenWrapper != null;
        File parent = new File(outputDirectory, ".mvn" + File.separator + "wrapper");
        parent.mkdirs();
        File file = new File(parent, MAVEN_WRAPPER_PROPERTIES);
        try (OutputStream mvnwOut = new FileOutputStream(file)) {
          mvnwOut.write(mavenWrapper.readAllBytes());
        }
      }

      try (InputStream mvnw = getClass().getClassLoader().getResourceAsStream(MVNW_CMD)) {
        assert mvnw != null;
        File file = new File(outputDirectory, MVNW_CMD);
        try (OutputStream mvnwOut = new FileOutputStream(file)) {
          mvnwOut.write(mvnw.readAllBytes());
          file.setExecutable(true);
        }
      }
    } catch (Exception e) {
      throw new RuntimeException(e);
    }
  }

  private void writeResource(String resourcePath, File outputFile) throws Exception {
    try (InputStream resourceAsStream = getClass().getClassLoader().getResourceAsStream(resourcePath)) {
      assert resourceAsStream != null;
      try (OutputStream outputStream = new FileOutputStream(outputFile)) {
        outputStream.write(resourceAsStream.readAllBytes());
      }
    }
  }

  public static void main(String[] args) {
    final MuleAppGenerator muleAppGenerator = new MuleAppGenerator();
    final String command = args[0];
    if (command.equalsIgnoreCase(GENERATE)) {
      String versions = args.length == 7 ? args[6] : "";
      Map<String, String> versionMap = Arrays.stream(versions.split(","))
          .filter(s -> !s.isBlank())
          .collect(Collectors.toMap(v -> {
            String[] split = v.split("=");
            if (split.length != 2) {
              throw new IllegalArgumentException("Invalid version `" + v + "` it needs to be with the shape of name=version");
            }
            return split[0];
          }, v -> {
            String[] split = v.split("=");
            if (split.length != 2) {
              throw new IllegalArgumentException("Invalid version `" + v + "` it needs to be with the shape of name=version");
            }
            return split[1];
          }));
      muleAppGenerator.generateMuleApp(
          new File(args[1]),
          new File(args[2]),
          Boolean.parseBoolean(args[3]),
          args[4],
          new File(args[5]),
          versionMap);
      System.exit(0);
    } else {
      throw new IllegalArgumentException("Usage: <generate> <mainFile> <exchangeJsonFile> <lowerEnv> <modulePath> <outputDirectory> <versions (name1=version1,name2=version2)>* ");
    }
  }
}
