/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.config;

import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.core.IsNull.nullValue;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.GatewaySecurityConfiguration;

import org.hamcrest.core.Is;
import org.junit.After;
import org.junit.BeforeClass;
import org.junit.Test;

public class GatewaySecurityConfigurationTestCase {

  private static final String ENCRYPTION_MODE = "anypoint.platform.policy_encryption_mode";
  private static final String ENCRYPTION_KEY = "anypoint.platform.encryption_key";

  private static final String HASH_CLIENTS = "anypoint.platform.hash_clients";
  private static final String HASH_CLIENTS_ALGORITHM = "anypoint.platform.hash_clients.algorithm";

  private static GatewaySecurityConfiguration configuration;

  @BeforeClass
  public static void setupPolicyEncryptionConfiguration() {
    configuration = new GatewayConfiguration().securityConfiguration();
  }

  @After
  public void clearEncryptionProperty() {
    clearProperty(ENCRYPTION_MODE);
    clearProperty(ENCRYPTION_KEY);
    clearProperty(HASH_CLIENTS);
    clearProperty(HASH_CLIENTS_ALGORITHM);
  }

  @Test
  public void defaultValueForEncryption() {
    assertThat(configuration.isSensitiveOnlyEncryption(), is(true));
  }

  @Test
  public void smartEncryptionEnabled() {
    setProperty(ENCRYPTION_MODE, "sensitive_only");
    assertThat(configuration.isSensitiveOnlyEncryption(), is(true));
  }

  @Test
  public void invalidConfigurationSetsToSmartEncryption() {
    setProperty(ENCRYPTION_MODE, "cualca");
    assertThat(configuration.isSensitiveOnlyEncryption(), is(true));
  }

  @Test
  public void differentCaseSetsToSmartEncryption() {
    setProperty(ENCRYPTION_MODE, "SensItiVe_OnLY");
    assertThat(configuration.isSensitiveOnlyEncryption(), is(true));
  }

  @Test
  public void fullEncryptionModeEnabled() {
    setProperty(ENCRYPTION_MODE, "full");
    assertThat(configuration.isSensitiveOnlyEncryption(), is(false));
  }

  @Test
  public void encryptionKey() {
    String dummyEncryptionKey = "ToInfinityAndBeyond";
    setProperty(ENCRYPTION_KEY, dummyEncryptionKey);
    assertThat(configuration.getEncryptionKey(), Is.is(dummyEncryptionKey));
  }

  @Test
  public void emptyEncryptionKey() {
    assertThat(configuration.getEncryptionKey(), nullValue());
  }

  @Test
  public void hashClientsFalseDefault() {
    assertThat(configuration.hashClients(), is(false));
  }

  @Test
  public void hashClientsEnabled() {
    setProperty(HASH_CLIENTS, "true");
    assertThat(configuration.hashClients(), is(true));
  }

  @Test
  public void hashClientsDefaultAlgorithm() {
    assertThat(configuration.hashAlgorithm(), is("SHA-256"));
  }

  @Test
  public void hashClientsDifferentAlgorithm() {
    setProperty(HASH_CLIENTS_ALGORITHM, "SHA-384");
    assertThat(configuration.hashAlgorithm(), is("SHA-384"));
  }

  @Test
  public void hashClientsDifferentAlgorithmLowerCase() {
    setProperty(HASH_CLIENTS_ALGORITHM, "sha-384");
    assertThat(configuration.hashAlgorithm(), is("SHA-384"));
  }

  @Test
  public void nonExistentHashingAlgorithmUsesDefault() {
    setProperty(HASH_CLIENTS_ALGORITHM, "SHA-UNKNOWN");
    assertThat(configuration.hashAlgorithm(), is("SHA-256"));
  }

}
