/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api.expirable.os;

import static com.google.common.collect.ImmutableMap.of;
import static java.util.Arrays.asList;
import static junit.framework.TestCase.fail;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.hamcrest.Matchers.is;
import static org.hamcrest.Matchers.not;
import static org.mockito.Mockito.mock;
import static org.mockito.Mockito.verify;
import static org.mockito.Mockito.verifyNoMoreInteractions;

import org.mule.runtime.api.store.ObjectDoesNotExistException;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;

import com.mulesoft.mule.runtime.gw.api.expirable.Expirable;
import com.mulesoft.mule.runtime.gw.api.expirable.mock.ControlledExpirableObjectFactory;
import com.mulesoft.mule.runtime.gw.api.expirable.mock.InMemoryObjectStore;

import java.util.HashMap;
import java.util.HashSet;

import org.hamcrest.MatcherAssert;
import org.junit.Before;
import org.junit.Test;

public class ExpirableObjectStoreTestCase {

  public static final String VALUE = "value";
  public static final String ANOTHER_VALUE = VALUE + "something else";
  public static final String KEY = "key";
  public static final String ANOTHER_KEY = "another-key";
  private ControlledExpirableObjectFactory<String> expirableObjectFactory;
  private InMemoryObjectStore<Expirable<String>> innerStore;
  private ExpirableObjectStore<String> os;

  @Before
  public void setUp() {
    expirableObjectFactory = new ControlledExpirableObjectFactory();
    innerStore = new InMemoryObjectStore<>();
    os = new ExpirableObjectStore<>(innerStore, expirableObjectFactory);
  }

  @Test
  public void containsMethod() throws ObjectStoreException {
    os.store(KEY, VALUE);
    assertThat(os.contains(KEY), is(true));
    assertThat(os.contains(ANOTHER_KEY), is(false));
  }

  @Test
  public void containsMethodOnAnExpiredEntry() throws ObjectStoreException {
    os.store(KEY, VALUE);
    expirableObjectFactory.expireAll();

    assertThat(os.contains(KEY), is(false));
  }

  @Test
  public void retrieveNonExpiredEntry() throws ObjectStoreException {
    os.store(KEY, VALUE);
    assertThat(os.retrieve(KEY), is(VALUE));
  }

  @Test
  public void expiredEntryThrowsOSExceptionAndRemovesEntry() throws ObjectStoreException {
    boolean exceptionRaised = false;
    os.store(KEY, VALUE);
    expirableObjectFactory.expireAll();

    try {
      os.retrieve(KEY);
    } catch (ObjectDoesNotExistException e) {
      assertThat(e.getMessage(), is("Value for " + KEY + " has been found but is expired."));
      MatcherAssert.assertThat(innerStore.contains(KEY), is(false));
      exceptionRaised = true;
    }

    if (!exceptionRaised) {
      fail("Exception should have been raised.");
    }
  }

  @Test
  public void removeNonExpiredEntry() throws ObjectStoreException {
    os.store(KEY, VALUE);
    assertThat(os.remove(KEY), is(VALUE));
    MatcherAssert.assertThat(innerStore.contains(KEY), is(false));
  }

  @Test
  public void removeExpiredEntry() throws ObjectStoreException {
    boolean exceptionRaised = false;
    os.store(KEY, VALUE);
    expirableObjectFactory.expireAll();

    try {
      os.remove(KEY);
    } catch (ObjectDoesNotExistException e) {
      assertThat(e.getMessage(), is("Value for " + KEY + " has been found but is expired."));
      MatcherAssert.assertThat(innerStore.contains(KEY), is(false));
      exceptionRaised = true;
    }

    if (!exceptionRaised) {
      fail("Exception should have been raised.");
    }
  }

  @Test
  public void clearAllEntries() throws ObjectStoreException {
    os.store(KEY, VALUE);
    os.store(ANOTHER_KEY, VALUE);
    os.clear();
    MatcherAssert.assertThat(innerStore.contains(KEY), is(false));
    MatcherAssert.assertThat(innerStore.contains(ANOTHER_KEY), is(false));
  }

  @Test
  public void storeRecreatesExpirable() throws ObjectStoreException {
    os.store(KEY, VALUE);
    Expirable<String> firstSavedObject = innerStore.retrieve(KEY);
    os.store(KEY, VALUE);
    Expirable<String> secondSavedObject = innerStore.retrieve(KEY);

    assertThat(firstSavedObject, not(secondSavedObject));
    assertThat(firstSavedObject.value(), is(VALUE));
    assertThat(secondSavedObject.value(), is(VALUE));
  }

  @Test
  public void openIsDelegated() throws ObjectStoreException {
    ObjectStore innerOs = mock(ObjectStore.class);
    os = new ExpirableObjectStore<>(innerOs, expirableObjectFactory);

    os.open();
    os.close();

    verify(innerOs).open();
    verify(innerOs).close();
    verifyNoMoreInteractions(innerOs);
  }

  @Test
  public void allKeysIsDelegated() throws ObjectStoreException {
    os.store(KEY, VALUE);
    os.store(ANOTHER_KEY, ANOTHER_VALUE);

    assertThat(new HashSet<>(os.allKeys()), is(new HashSet(asList(KEY, ANOTHER_KEY))));
  }

  @Test
  public void retrieveAll() throws ObjectStoreException {
    os.store(KEY, VALUE);
    os.store(ANOTHER_KEY, ANOTHER_VALUE);

    assertThat(os.retrieveAll(), is(of(KEY, VALUE, ANOTHER_KEY, ANOTHER_VALUE)));
  }

  @Test
  public void retrieveAllFilteringExpiredKeys() throws ObjectStoreException {
    os.store(KEY, VALUE);
    os.store(ANOTHER_KEY, ANOTHER_VALUE);
    expirableObjectFactory.expireAll();

    assertThat(os.retrieveAll(), is(new HashMap<>()));
  }
}
