/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.config;

import static com.mulesoft.mule.runtime.gw.api.time.period.Period.millis;
import static com.mulesoft.mule.runtime.gw.api.time.period.Period.seconds;
import static java.lang.System.setProperty;
import static java.util.Arrays.asList;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration;
import com.mulesoft.mule.runtime.gw.api.exception.GatewayConfigurationException;
import com.mulesoft.mule.runtime.gw.internal.encryption.GatewayEncryptionException;

import java.util.List;

import org.junit.After;
import org.junit.Test;

public class PlatformClientConfigurationTestCase {

  private PlatformClientConfiguration configuration = new GatewayConfiguration().platformClient();

  private List<String> systemProperties = asList("anypoint.platform.analytics_base_uri",
                                                 "anypoint.platform.client_id",
                                                 "anypoint.platform.client_secret",
                                                 "anypoint.platform.on_prem",
                                                 "anypoint.platform.base_uri",
                                                 "anypoint.platform.poll_policies_freq",
                                                 "anypoint.platform.poll_clients_freq",
                                                 "anypoint.platform.api_keep_alive_freq",
                                                 "anypoint.platform.proxy_host",
                                                 "anypoint.platform.proxy_port",
                                                 "anypoint.platform.proxy_username",
                                                 "anypoint.platform.proxy_password",
                                                 "anypoint.platform.outage_status_codes",
                                                 "anypoint.platform.connect_timeout",
                                                 "anypoint.platform.read_timeout",
                                                 "anypoint.platform.enable_ssl_validation",
                                                 "anypoint.platform.policy_template_source",
                                                 "anypoint.platform.metrics.base_uri",
                                                 "anypoint.platform.metrics_base_uri");

  @After
  public void tearDown() {
    systemProperties.forEach(System::clearProperty);
  }

  @Test
  public void connectTimeoutDefault() {
    assertEquals(configuration.getConnectTimeout(), millis(10000));
  }

  @Test
  public void connectTimeoutCustom() {
    setProperty("anypoint.platform.connect_timeout", "30000");
    assertEquals(configuration.getConnectTimeout(), millis(30000));
  }

  @Test
  public void connectTimeoutCustomForever() {
    setProperty("anypoint.platform.connect_timeout", "0");
    assertEquals(configuration.getConnectTimeout(), millis(0));
  }

  @Test
  public void connectTimeoutInvalid() {
    setProperty("anypoint.platform.connect_timeout", "alpha");
    assertEquals(configuration.getConnectTimeout(), millis(10000));
    setProperty("anypoint.platform.connect_timeout", "-1000");
    assertEquals(configuration.getConnectTimeout(), millis(10000));
  }

  @Test
  public void readTimeoutDefault() {
    assertEquals(configuration.getReadTimeout(), millis(10000));
  }

  @Test
  public void readTimeoutCustom() {
    setProperty("anypoint.platform.read_timeout", "60000");
    assertEquals(configuration.getReadTimeout(), millis(60000));
  }

  @Test
  public void readTimeoutCustomForever() {
    setProperty("anypoint.platform.read_timeout", "0");
    assertEquals(configuration.getReadTimeout(), millis(0));
  }

  @Test
  public void readTimeoutInvalid() {
    setProperty("anypoint.platform.read_timeout", "alpha");
    assertEquals(configuration.getReadTimeout(), millis(10000));
    setProperty("anypoint.platform.read_timeout", "-1000");
    assertEquals(configuration.getReadTimeout(), millis(10000));
  }

  @Test
  public void policiesPollFrequencyDefault() {
    assertEquals(configuration.getApisPollFrequency(), seconds(60));
  }

  @Test
  public void policiesPollFrequencyInvalid() {
    setProperty("anypoint.platform.poll_policies_freq", "invalid");
    assertEquals(configuration.getApisPollFrequency(), seconds(60));
  }

  @Test
  public void policiesPollFrequencyPropertyValue() {
    setProperty("anypoint.platform.poll_policies_freq", "30");
    assertEquals(configuration.getApisPollFrequency(), seconds(30));
  }

  @Test
  public void keepAliveFrequencyDefault() {
    assertEquals(configuration.getApiKeepAliveFrequency(), seconds(60));
  }

  @Test
  public void keepAliveFrequencyInvalid() {
    setProperty("anypoint.platform.api_keep_alive_freq", "invalid");
    assertEquals(configuration.getApiKeepAliveFrequency(), seconds(60));
  }

  @Test
  public void keepAliveFrequencyPropertyValue() {
    setProperty("anypoint.platform.api_keep_alive_freq", "30");
    assertEquals(configuration.getApiKeepAliveFrequency(), seconds(30));
  }

  @Test
  public void clientsPollFrequencyDefault() {
    assertEquals(configuration.getClientsPollFrequency(), seconds(15));
  }

  @Test
  public void clientsPollFrequencyInvalid() {
    setProperty("anypoint.platform.poll_clients_freq", "invalid");
    assertEquals(configuration.getClientsPollFrequency(), seconds(15));
  }

  @Test
  public void clientsPollFrequencyPropertyValue() {
    setProperty("anypoint.platform.poll_clients_freq", "30");
    assertEquals(configuration.getClientsPollFrequency(), seconds(30));
  }

  @Test
  public void platformUriDefault() {
    assertThat(configuration.getPlatformUri(), is("https://anypoint.mulesoft.com"));
  }

  @Test
  public void platformUri() {
    String platformBaseUri = "https://platform-base-uri.com";
    setProperty("anypoint.platform.base_uri", platformBaseUri);
    assertThat(configuration.getPlatformUri(), is(platformBaseUri));
  }

  @Test
  public void clientId() {
    String clientId = "clientId";
    setProperty("anypoint.platform.client_id", clientId);
    assertThat(configuration.getClientId(), is(clientId));
  }

  @Test(expected = GatewayConfigurationException.class)
  public void encryptedClientIdWihMissingEncryptionKey() {
    String clientId = "![clientId]";
    setProperty("anypoint.platform.client_id", clientId);
    configuration.getClientId();
  }

  @Test
  public void clientSecret() {
    String clientSecret = "clientSecret";
    setProperty("anypoint.platform.client_secret", clientSecret);
    assertThat(configuration.getClientSecret(), is(clientSecret));
  }

  @Test(expected = GatewayConfigurationException.class)
  public void encryptedClientSecretWithMissingEncryptionKey() throws GatewayEncryptionException {
    String clientSecret = "![clientSecret]";
    setProperty("anypoint.platform.client_secret", clientSecret);
    configuration.getClientSecret();
  }

  @Test
  public void offlineMode() {
    assertThat(configuration.isOfflineModeEnabled(), is(true));
  }

  @Test
  public void onlineMode() {
    setProperty("anypoint.platform.client_id", "clientId");
    setProperty("anypoint.platform.client_secret", "clientSecret");
    assertThat(configuration.isOfflineModeEnabled(), is(false));
  }

  @Test
  public void onPrem() {
    setProperty("anypoint.platform.on_prem", "true");
    assertThat(configuration.isOnPrem(), is(true));
  }

  @Test
  public void proxyHost() {
    String proxyHost = "localhost";
    setProperty("anypoint.platform.proxy_host", proxyHost);
    assertThat(configuration.getProxyHost(), is(proxyHost));
  }

  @Test
  public void proxyPortDefault() {
    assertThat(configuration.getProxyPort(), is(80));
  }

  @Test
  public void proxyPort() {
    setProperty("anypoint.platform.proxy_port", "8080");
    assertThat(configuration.getProxyPort(), is(8080));
  }

  @Test
  public void proxyUsername() {
    String proxyUsername = "username";
    setProperty("anypoint.platform.proxy_username", proxyUsername);
    assertThat(configuration.getProxyUserName(), is(proxyUsername));
  }

  @Test
  public void proxyPassword() {
    String proxyPassword = "password";
    setProperty("anypoint.platform.proxy_password", proxyPassword);
    assertThat(configuration.getProxyPassword(), is(proxyPassword));
  }

  @Test(expected = GatewayConfigurationException.class)
  public void encryptedProxyPasswordWithMissingEncryptionKey() throws GatewayEncryptionException {
    String proxyPassword = "![password]";
    setProperty("anypoint.platform.proxy_password", proxyPassword);
    configuration.getProxyPassword();
  }

  @Test
  public void defaultOutageStatusCodes() {
    assertThat(configuration.getOutagesStatusCodes(), is(asList(400, 401, 403, 405, 429, 502, 503)));
  }

  @Test
  public void setOutageStatusCodes() {
    setProperty("anypoint.platform.outage_status_codes", "666, 207");

    assertThat(configuration.getOutagesStatusCodes(), is(asList(666, 207)));
  }

  @Test
  public void emptyOutageStatusCodes() {
    setProperty("anypoint.platform.outage_status_codes", "");

    assertThat(configuration.getOutagesStatusCodes(), is(asList(400, 401, 403, 405, 429, 502, 503)));
  }

  @Test
  public void illFormedOutageStatusCodes() {
    setProperty("anypoint.platform.outage_status_codes", "534f");

    assertThat(configuration.getOutagesStatusCodes(), is(asList(400, 401, 403, 405, 429, 502, 503)));
  }

  @Test
  public void outageStatusCodesFiltersSpaces() {
    setProperty("anypoint.platform.outage_status_codes", "   666 ,   207 ");

    assertThat(configuration.getOutagesStatusCodes(), is(asList(666, 207)));
  }

  @Test
  public void emptyStatusCode() {
    setProperty("anypoint.platform.outage_status_codes", "534,,666");

    assertThat(configuration.getOutagesStatusCodes(), is(asList(400, 401, 403, 405, 429, 502, 503)));
  }

  @Test
  public void SSLValidationIsEnabledByDefault() {
    assertThat(configuration.enableSSLValidation(), is(true));
  }

  @Test
  public void disableSSLValidation() {
    setProperty("anypoint.platform.enable_ssl_validation", "false");

    assertThat(configuration.enableSSLValidation(), is(false));
  }

  @Test
  public void policyTemplateSourceDefault() {
    assertThat(configuration.getPolicyTemplateSource(), is("internal"));
  }

  @Test
  public void metricsDefaultUri() {
    assertThat(configuration.getMetricsBaseUri(), is("https://anypoint.mulesoft.com"));
  }

  @Test
  public void metricsLegacyCustomUri() {
    setProperty("anypoint.platform.metrics.base_uri", "https://example.com/foo");
    assertThat(configuration.getMetricsBaseUri(), is("https://example.com/foo"));
  }

  @Test
  public void metricsCustomUri() {
    setProperty("anypoint.platform.metrics_base_uri", "https://example.com/foo");
    assertThat(configuration.getMetricsBaseUri(), is("https://example.com/foo"));
  }
}
