/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api.policy;

import static java.util.regex.Pattern.CASE_INSENSITIVE;

import org.mule.runtime.api.component.ComponentIdentifier;
import org.mule.runtime.http.policy.api.HttpPolicyPointcutParameters;
import org.mule.runtime.http.policy.api.SourcePolicyAwareAttributes;
import org.mule.runtime.policy.api.AttributeAwarePointcut;
import org.mule.runtime.policy.api.PolicyAwareAttributes;
import org.mule.runtime.policy.api.PolicyPointcutParameters;

import java.io.Serializable;
import java.util.regex.Pattern;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class HttpResourcePointcut implements AttributeAwarePointcut, Serializable {

  public static final ComponentIdentifier HTTP_REQUEST_COMPONENT_IDENTIFIER =
      ComponentIdentifier.builder().namespace("http").name("request").build();
  public static final ComponentIdentifier HTTP_LISTENER_COMPONENT_IDENTIFIER =
      ComponentIdentifier.builder().namespace("http").name("listener").build();
  private static final Logger LOGGER = LoggerFactory.getLogger(HttpResourcePointcut.class);
  private static final long serialVersionUID = -5003310196710218800L;

  /**
   * Regular expression to match against the HTTP request's path
   */
  private final Pattern pathPattern;

  /**
   * Regular expression to match against the HTTP request's method
   */
  private final Pattern methodPattern;

  public HttpResourcePointcut(String pathRegex, String methodRegex) {
    this.pathPattern = Pattern.compile(pathRegex, CASE_INSENSITIVE);
    this.methodPattern = Pattern.compile(methodRegex, CASE_INSENSITIVE);
  }

  @Override
  public boolean matches(PolicyPointcutParameters policyPointcutParameters) {
    PolicyPointcutParameters parameters = policyPointcutParameters.getSourceParameters().orElse(policyPointcutParameters);

    try {
      String path = ((HttpPolicyPointcutParameters) parameters).getPath();
      String method = ((HttpPolicyPointcutParameters) parameters).getMethod();

      boolean matches = methodPattern.matcher(method).matches() && pathPattern.matcher(path).matches();

      if (!matches) {
        LOGGER.trace("Request {{}, {}} does not match the following resource pointcut: {}", path, method, this);
      } else {
        LOGGER.trace("Request {{}, {}} matches the following resource pointcut: {}", path, method, this);
      }

      return matches;
    } catch (ClassCastException e) {
      LOGGER.trace("Invalid parameters type found. {}", e.getMessage());
    }

    return false;
  }

  @Override
  public PolicyAwareAttributes sourcePolicyAwareAttributes() {
    return new SourcePolicyAwareAttributes.Builder().requestPathPatterns(pathPattern).build();
  }

  public String getPath() {
    return pathPattern.toString();
  }

  public String getMethod() {
    return methodPattern.toString();
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }

    HttpResourcePointcut that = (HttpResourcePointcut) o;

    return pathPattern.toString().equals(that.pathPattern.toString()) && methodPattern.toString()
        .equals(that.methodPattern.toString());
  }

  @Override
  public int hashCode() {
    int result = pathPattern.toString().hashCode();
    result = 31 * result + (methodPattern.toString().hashCode());
    return result;
  }

  @Override
  public String toString() {
    return "{path=" + pathPattern + ", method=" + methodPattern + "}";
  }
}
