/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api;

import org.mule.runtime.api.lifecycle.Disposable;
import org.mule.runtime.api.lifecycle.Initialisable;

import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.api.contract.Contract;
import com.mulesoft.mule.runtime.gw.api.contract.Sla;
import com.mulesoft.mule.runtime.gw.api.exception.ForbiddenClientException;

import java.util.ArrayList;
import java.util.List;

/**
 * Defines an API's related {@link Contract}s. This class will remain backwards compatible.
 *
 * @since 4.0
 */
public interface ApiContracts extends Disposable, Initialisable {

  /**
   * Updates the available {@link Contract}s.
   *
   * @param contracts current {@link Contract}s.
   * @return (Subclass) this.
   */
  ApiContracts updateContracts(List<Contract> contracts);

  /**
   * Updates the available {@link Sla}s.
   *
   * @param slas current {@link Sla}s.
   * @return (Subclass) this.
   */
  default ApiContracts updateSlas(List<Sla> slas) {
    return this;
  }

  /**
   * Updates the available {@link Contract}s.
   *
   * @param contracts current {@link Contract}s.
   * @return (Subclass) this.
   */
  ApiContracts updateContracts(Contract... contracts);

  /**
   * @return the SLA of the Client, if there is no Contract for that client, an exception will be raised
   *
   * @throws ForbiddenClientException when the client has no SLA defined
   */
  Sla sla(Client client) throws ForbiddenClientException;

  /**
   * @return the SLA for a client id, if there is no Contract for that id, an exception will be raised
   *
   * @throws ForbiddenClientException when the client has no SLA defined
   */
  Sla sla(String clientId) throws ForbiddenClientException;

  /**
   * Validates that the {@link Client} is within a {@link Contract} for this Api, otherwise throws an exception.
   *
   * @return this
   * @throws ForbiddenClientException when the client has no contract defined in the API
   */
  Client validate(String clientId, String clientSecret) throws ForbiddenClientException;

  /**
   * Validates that the {@link Client} is within a {@link Contract} for this API, otherwise throws an exception.
   *
   * @return this
   * @throws ForbiddenClientException when the client has no contract defined in the API
   */
  Client validate(String clientId) throws ForbiddenClientException;

  /**
   * @return every {@link Contract} related to the specified API
   */
  default List<Contract> contracts() {
    return new ArrayList<>();
  }

  /**
   * Indicates if contracts for the API were successfully loaded from the Disk, Platform, etc.
   */
  default boolean contractsLoaded() {
    return true;
  }

  /**
   * {@inheritDoc}
   */
  default void initialise() {}

  /**
   * {@inheritDoc}
   */
  default void dispose() {}
}
