/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api.config;

import static com.mulesoft.mule.runtime.gw.api.config.GatewaySecurityConfiguration.EncryptionMode.SENSITIVE_ONLY;
import static java.lang.System.clearProperty;
import static java.lang.System.getProperty;
import static org.apache.commons.lang3.StringUtils.trimToNull;

import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;

public class GatewaySecurityConfiguration extends RuntimeConfiguration {

  /**
   * Used to encrypt/decrypt sensitive information for policies and GW configuration
   */
  private static final String ENCRYPTION_KEY = "anypoint.platform.encryption_key";

  /**
   * Used to encrypt/decrypt sensitive information for policies and GW configuration
   */
  private static final String POLICY_ENCRYPTION_MODE = "anypoint.platform.policy_encryption_mode";
  private static final EncryptionMode POLICY_ENCRYPTION_MODE_DEFAULT = SENSITIVE_ONLY;

  /**
   * This flag enables or disables hashing {@link com.mulesoft.mule.runtime.gw.api.client.Client} information. Default is true.
   */
  private static final String HASH_CLIENTS = "anypoint.platform.hash_clients";
  private static final String HASH_CLIENTS_DEFAULT = "true";

  /**
   * This flag determines the algorithm to hash {@link com.mulesoft.mule.runtime.gw.api.client.Client} information. Default is
   * SHA-256.
   */
  private static final String HASH_CLIENTS_ALGORITHM = "anypoint.platform.hash_clients.algorithm";
  private static final String HASH_CLIENTS_ALGORITHM_DEFAULT = "SHA-256";

  public String getEncryptionKey() {
    return trimToNull(getProperty(ENCRYPTION_KEY));
  }

  public boolean isEncryptionEnabled() {
    return getEncryptionKey() != null;
  }

  public boolean isSensitiveOnlyEncryption() {
    return parsePropertyValue().equals(SENSITIVE_ONLY);
  }

  private EncryptionMode parsePropertyValue() {
    String modeInput = trimToNull(getProperty(POLICY_ENCRYPTION_MODE, POLICY_ENCRYPTION_MODE_DEFAULT.name()));

    try {
      return EncryptionMode.valueOf(modeInput.toUpperCase());
    } catch (IllegalArgumentException e) {
      LOGGER.warn("{} is not a valid Policy Encryption configuration mode. {} mode will be used", modeInput,
                  POLICY_ENCRYPTION_MODE_DEFAULT);
      return POLICY_ENCRYPTION_MODE_DEFAULT;
    }
  }

  public void clearEncryptionKey() {
    clearProperty(ENCRYPTION_KEY);
  }

  public enum EncryptionMode {
    FULL, SENSITIVE_ONLY
  }

  public boolean hashClients() {
    return Boolean.valueOf(trimToNull(getProperty(HASH_CLIENTS, HASH_CLIENTS_DEFAULT)));
  }

  public String hashAlgorithm() {
    String algorithm = trimToNull(getProperty(HASH_CLIENTS_ALGORITHM, HASH_CLIENTS_ALGORITHM_DEFAULT)).toUpperCase();
    try {
      MessageDigest.getInstance(algorithm);
    } catch (NoSuchAlgorithmException e) {
      LOGGER.warn("{} is not a valid Hashing algorithm. {} algorithm will be used.", algorithm, HASH_CLIENTS_ALGORITHM_DEFAULT);
      return HASH_CLIENTS_ALGORITHM_DEFAULT;
    }
    return algorithm;
  }

}
