/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api.expirable.os;

import static com.mulesoft.mule.runtime.gw.api.functional.StreamsInMaps.mapWithValues;
import static java.lang.String.format;

import org.mule.runtime.api.store.ObjectDoesNotExistException;
import org.mule.runtime.api.store.ObjectStore;
import org.mule.runtime.api.store.ObjectStoreException;

import com.mulesoft.mule.runtime.gw.api.expirable.Expirable;
import com.mulesoft.mule.runtime.gw.api.expirable.ExpirableObjectFactory;
import com.mulesoft.mule.runtime.gw.api.expirable.exception.ExpirableObjectMessage;

import java.io.Serializable;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Map.Entry;
import java.util.Set;
import java.util.function.Predicate;

/**
 * Decorates an {@link ObjectStore}, ensuring that when any message is answered, the requested entry has not yet expired.
 * <p>
 * NOTE (I): Mule API already provides an {@link org.mule.runtime.api.store.ExpirableObjectStore}. This implementation differs
 * from that in that each entry can be expired at a different pace by delegating the expiration configuration in the
 * {@link ExpirableObjectFactory}. NOTE (II): This implementation is NOT thread safe.
 *
 * @param <T> type of entry.
 */
public class ExpirableObjectStore<T extends Serializable> implements ObjectStore<T> {

  private final ObjectStore<Expirable<T>> innerOS;
  private final ExpirableObjectFactory<T> expirableObjectFactory;

  public ExpirableObjectStore(ObjectStore<Expirable<T>> innerOS, ExpirableObjectFactory<T> expirableObjectFactory) {
    this.innerOS = innerOS;
    this.expirableObjectFactory = expirableObjectFactory;
  }

  /***
   * {@inheritDoc}
   * <p>
   * If entry exists but has expired, this method will return false.
   */
  @Override
  public boolean contains(String key) throws ObjectStoreException {
    return innerOS.contains(key) && !innerOS.retrieve(key).isExpired();
  }

  /**
   * {@inheritDoc}
   * <p>
   * Storing multiple times values for the same key will result in overwriting the previous values.
   */
  @Override
  public void store(String key, T value) throws ObjectStoreException {
    if (innerOS.contains(key)) {
      innerOS.remove(key);
    }
    innerOS.store(key, expirableObjectFactory.create(value));
  }

  /**
   * {@inheritDoc}
   * <p>
   * If the entry exists but has expired, it will be removed from is internal {@link ObjectStore} and an
   * {@link ObjectDoesNotExistException} will be thrown.
   */
  @Override
  public T retrieve(String key) throws ObjectStoreException {
    Expirable<T> retrievedValue = innerOS.retrieve(key);
    if (retrievedValue.isExpired()) {
      innerOS.remove(key);
      throw expiredEntryException(key);
    }
    return retrievedValue.value();
  }

  /**
   * {@inheritDoc}
   * <p>
   * Entry will be always removed, but if it has expired, an {@link ObjectDoesNotExistException} will be thrown.
   */
  @Override
  public T remove(String key) throws ObjectStoreException {
    Expirable<T> removedExpirable = innerOS.remove(key);
    if (removedExpirable.isExpired()) {
      throw expiredEntryException(key);
    }
    return removedExpirable.value();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public boolean isPersistent() {
    return innerOS.isPersistent();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void clear() throws ObjectStoreException {
    innerOS.clear();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void open() throws ObjectStoreException {
    innerOS.open();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public void close() throws ObjectStoreException {
    innerOS.close();
  }

  /**
   * {@inheritDoc}
   */
  @Override
  public List<String> allKeys() throws ObjectStoreException {
    return innerOS.allKeys();
  }

  /**
   * {@inheritDoc}
   * <p>
   * As every entry must be processed, this {@link ObjectStore} will clean up expired entries before returning them. Beware that
   * an entry can expire between when is processed and this method returns.
   *
   * @return All non-expired objects.
   */
  @Override
  public Map<String, T> retrieveAll() throws ObjectStoreException {
    Set<String> keysToRemove = new HashSet<>();

    Map<String, T> nonExpiredEntries = innerOS.retrieveAll().entrySet().stream()
        .filter(expiredEntries(keysToRemove))
        .collect(mapWithValues(e -> e.getValue().value()));

    cleanUpInnerOs(keysToRemove);

    return nonExpiredEntries;
  }

  private void cleanUpInnerOs(Set<String> keysToRemove) throws ObjectStoreException {
    for (String key : keysToRemove) {
      innerOS.remove(key);
    }
  }

  private Predicate<Entry<String, Expirable<T>>> expiredEntries(Set<String> keysToRemove) {
    return e -> {
      boolean expiredEntry = e.getValue().isExpired();
      if (expiredEntry) {
        keysToRemove.add(e.getKey());
      }
      return !expiredEntry;
    };
  }

  protected ObjectDoesNotExistException expiredEntryException(Serializable key) {
    return new ObjectDoesNotExistException(
                                           new ExpirableObjectMessage(format("Value for %s has been found but is expired.",
                                                                             key)));
  }
}
