/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api.contract;

import static java.util.Objects.requireNonNull;

import com.mulesoft.mule.runtime.gw.api.client.Client;
import com.mulesoft.mule.runtime.gw.api.construction.Builder;

import java.io.Serializable;

/**
 * Represents the relationship between a Client and an SLA. This will usually be used in the context of an Api
 */
public class Contract implements Serializable {

  private static final long serialVersionUID = 2798205646171486620L;
  private final Sla sla;
  private final Client client;

  private Contract(Client client, Sla sla) {
    this.sla = sla;
    this.client = client;
  }

  public static ContractBuilder builder() {
    return new ContractBuilder();
  }

  /**
   * @return the Contract's Client
   */
  public Client client() {
    return client;
  }

  /**
   * @return the Contract's Sla
   */
  public Sla sla() {
    return sla;
  }

  @Override
  public String toString() {
    return "Contract{" +
        "sla=" + sla +
        ", client=" + client +
        '}';
  }

  @Override
  public boolean equals(Object o) {
    if (this == o) {
      return true;
    }
    if (!(o instanceof Contract)) {
      return false;
    }

    Contract that = (Contract) o;

    if (!sla.equals(that.sla)) {
      return false;
    }
    return client.equals(that.client);
  }

  @Override
  public int hashCode() {
    int result = sla.hashCode();
    result = 31 * result + client.hashCode();
    return result;
  }

  public static class ContractBuilder implements Builder<Contract> {

    private Sla sla;
    private Client client;

    public ContractBuilder() {
      this.sla = new NoSla();
    }

    public ContractBuilder withSla(Sla sla) {
      this.sla = sla;
      return this;
    }

    public ContractBuilder withClient(Client client) {
      this.client = client;
      return this;
    }

    @Override
    public Contract build() {
      requireNonNull(sla, "Cannot create Contract without an SLA");
      requireNonNull(client, "Cannot create Contract without a Client");
      return new Contract(client, sla);
    }
  }
}
