/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api.time.clock;

import com.mulesoft.mule.runtime.gw.api.time.DateTime;
import com.mulesoft.mule.runtime.gw.internal.time.clock.SystemClock;

import java.io.Serializable;

/**
 * Defines a Clock where its time can be corrected.
 */
public abstract class Clock implements Serializable {

  private static final long A_SECOND = 1000;
  private static final long serialVersionUID = 2343112458335437523L;
  private static Clock atomicInstance;
  protected long millisecondsOffset = 0;

  public DateTime now() {
    return currentTime().plus(millisecondsOffset);
  }

  protected abstract DateTime currentTime();

  public Clock correctWith(Clock atomicClock) {
    millisecondsOffset = atomicClock.now().getMillis() - this.currentTime().getMillis();
    return this;
  }

  @Override
  public boolean equals(Object anotherObject) {
    if (anotherObject == null || !(anotherObject instanceof Clock)) {
      return false;
    }

    DateTime thisNow = this.now();
    DateTime otherNow = ((Clock) anotherObject).now();

    long millisDiff = Math.abs(thisNow.getMillis() - otherNow.getMillis());
    return millisDiff < A_SECOND;
  }

  @Override
  public String toString() {
    return "[now: " + now() + ", clockCorrection(ms): " + millisecondsOffset + "]";
  }

  public static void setAtomic(Clock clock) {
    /**
     * We only need a static instance of the clock to resolve the CurrentTimeInMillis callable when running on Hazelcast
     */
    atomicInstance = clock;
  }

  public static Clock atomic() {
    return atomicInstance;
  }

  public static Clock system() {
    return new SystemClock();
  }

  public long offset() {
    return millisecondsOffset;
  }
}
