/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.internal.time.frame;

import com.mulesoft.mule.runtime.gw.api.time.DateTime;
import com.mulesoft.mule.runtime.gw.api.time.clock.Clock;
import com.mulesoft.mule.runtime.gw.api.time.frame.TimeFrame;
import com.mulesoft.mule.runtime.gw.api.time.period.Period;

/**
 * Defines a fixed time frame.
 * <p/>
 * Knows his bounds, a given a clock knows if it is current and who is the next valid TimeFrame (could be itself if frame is still
 * current).
 */
public class FixedTimeFrame implements TimeFrame {

  private static final long serialVersionUID = -2098663731137982314L;
  private final Clock clock;
  private final Period period;
  private final DateTime startTime;
  private final DateTime endTime;

  public FixedTimeFrame(Clock clock, DateTime startTime, Period period) {
    this.clock = clock;
    this.period = period;
    this.startTime = startTime;
    this.endTime = startTime.plus(period.inMillis() - 1);
  }

  @Override
  public boolean hasFinished() {
    DateTime now = clock.now();
    return !(inRange(now) || isBorder(now));
  }

  private boolean isBorder(DateTime time) {
    return time.equals(startTime) || time.equals(endTime);
  }

  protected boolean inRange(DateTime time) {
    return time.isAfter(startTime) && time.isBefore(endTime);
  }


  @Override
  public long millisUntilNextTimeFrame() {
    long remainingTime = endTime.getMillis() - clock.now().getMillis();

    return remainingTime > 0 ? remainingTime : 0;
  }

  @Override
  public long inMillis() {
    return period.inMillis();
  }

  @Override
  public TimeFrame next() {
    if (!hasFinished()) {
      return this;
    }

    return new FixedTimeFrame(clock, nextStartDate(), period);
  }

  @Override
  public DateTime startTime() {
    return startTime;
  }

  @Override
  public boolean contains(DateTime dateTime) {
    return inRange(dateTime) || isBorder(dateTime);
  }

  private DateTime nextStartDate() {
    DateTime nextStartDate;
    long diffStartAndNow = clock.now().getMillis() - startTime.getMillis();
    long startOffset = Math.abs(diffStartAndNow) % period.inMillis();
    if (diffStartAndNow >= 0) {
      nextStartDate = startTime.plus(diffStartAndNow - startOffset);
    } else {
      nextStartDate = startTime.minus(Math.abs(diffStartAndNow) + period.inMillis() - startOffset);
    }

    return nextStartDate;
  }

  @Override
  public boolean equals(Object anotherObject) {
    if (anotherObject == null || !(anotherObject instanceof FixedTimeFrame)) {
      return false;
    }

    FixedTimeFrame anotherFrame = (FixedTimeFrame) anotherObject;

    return this.startTime.equals(anotherFrame.startTime) &&
        this.endTime.equals(anotherFrame.endTime) &&
        this.period.equals(anotherFrame.period);
  }

  @Override
  public String toString() {
    return "FixedTimeFrame{ start:" + startTime +
        ", end:" + endTime +
        ", duration:" + period.inMillis() +
        ", remainingFrame:" + millisUntilNextTimeFrame() +
        '}';
  }
}
