/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.config;

import static com.mulesoft.mule.runtime.gw.api.time.period.Period.millis;
import static com.mulesoft.mule.runtime.gw.api.time.period.Period.seconds;
import static java.lang.System.clearProperty;
import static java.lang.System.setProperty;
import static java.util.Arrays.asList;
import static java.util.Optional.empty;
import static java.util.Optional.of;
import static org.hamcrest.core.Is.is;
import static org.junit.Assert.assertEquals;
import static org.junit.Assert.assertThat;

import com.mulesoft.mule.runtime.gw.api.config.GatewayConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.PlatformClientConfiguration;
import com.mulesoft.mule.runtime.gw.api.config.PlatformServicesConfiguration;
import com.mulesoft.mule.runtime.gw.api.exception.GatewayConfigurationException;
import com.mulesoft.mule.runtime.gw.internal.encryption.GatewayEncryptionException;

import java.io.File;
import java.util.List;
import java.util.function.Supplier;

import org.junit.After;
import org.junit.Test;

public class PlatformServicesConfigurationTestCase {

  public static final String POLL_POLICIES_ENABLED_KEY = "anypoint.platform.poll_policies_enabled";
  public static final String POLL_CLIENTS_ENABLED_KEY = "anypoint.platform.poll_clients_enabled";
  public static final String KEEP_ALIVE_ENABLED_KEY = "anypoint.platform.api_keep_alive_enabled";
  public static final String ANYPOINT_PLATFORM_APIS_LOCATION = "anypoint.platform.apis_location";
  private PlatformServicesConfiguration configuration = new GatewayConfiguration().platformServices();

  private List<String> systemProperties = asList(POLL_POLICIES_ENABLED_KEY,
                                                 POLL_POLICIES_ENABLED_KEY,
                                                 KEEP_ALIVE_ENABLED_KEY,
                                                 ANYPOINT_PLATFORM_APIS_LOCATION);

  @After
  public void tearDown() {
    systemProperties.forEach(System::clearProperty);
  }

  @Test
  public void apiPollEnabledPropertyValueNotDefined() {
    undefinedPropertyReturnsDefault(configuration::getApisPollEnabled, POLL_POLICIES_ENABLED_KEY, true);
  }

  @Test
  public void apiPollEnabledPropertyValueTrue() {
    propertyWithValueReturnsExpected(configuration::getApisPollEnabled, POLL_POLICIES_ENABLED_KEY, "true", true);
  }

  @Test
  public void apiPollEnabledPropertyValueFalse() {
    propertyWithValueReturnsExpected(configuration::getApisPollEnabled, POLL_POLICIES_ENABLED_KEY, "false", false);
  }

  @Test
  public void apiPollEnabledPropertyValueInvalid() {
    propertyWithValueReturnsExpected(configuration::getApisPollEnabled, POLL_POLICIES_ENABLED_KEY, "foo", false);
  }

  @Test
  public void clientsPollEnabledPropertyValueNotDefined() {
    undefinedPropertyReturnsDefault(configuration::getClientsPollEnabled, POLL_CLIENTS_ENABLED_KEY, true);
  }

  @Test
  public void clientsPollEnabledPropertyValueTrue() {
    propertyWithValueReturnsExpected(configuration::getClientsPollEnabled, POLL_CLIENTS_ENABLED_KEY, "true", true);
  }

  @Test
  public void clientsPollEnabledPropertyValueFalse() {
    propertyWithValueReturnsExpected(configuration::getClientsPollEnabled, POLL_CLIENTS_ENABLED_KEY, "false", false);
  }

  @Test
  public void clientsPollEnabledPropertyValueInvalid() {
    propertyWithValueReturnsExpected(configuration::getClientsPollEnabled, POLL_CLIENTS_ENABLED_KEY, "foo", false);
  }

  @Test
  public void keepAliveEnabledPropertyValueNotDefined() {
    undefinedPropertyReturnsDefault(configuration::getApiKeepAliveEnabled, KEEP_ALIVE_ENABLED_KEY, true);
  }

  @Test
  public void keepAliveEnabledPropertyValueTrue() {
    propertyWithValueReturnsExpected(configuration::getApiKeepAliveEnabled, KEEP_ALIVE_ENABLED_KEY, "true", true);
  }

  @Test
  public void keepAliveEnabledPropertyValueFalse() {
    propertyWithValueReturnsExpected(configuration::getApiKeepAliveEnabled, KEEP_ALIVE_ENABLED_KEY, "false", false);
  }

  @Test
  public void keepAliveEnabledPropertyValueInvalid() {
    propertyWithValueReturnsExpected(configuration::getApiKeepAliveEnabled, KEEP_ALIVE_ENABLED_KEY, "foo", false);
  }

  @Test
  public void apisFolderSuccessfullyFound() {
    setProperty(ANYPOINT_PLATFORM_APIS_LOCATION, "foo");

    assertThat(configuration.apisFolder(), is(of(new File("foo"))));
  }

  @Test
  public void nonExistentApisFolder() {
    setProperty(ANYPOINT_PLATFORM_APIS_LOCATION, "");

    assertThat(configuration.apisFolder(), is(empty()));
  }

  private void undefinedPropertyReturnsDefault(Supplier<Boolean> supplier, String key, boolean defaultValue) {
    clearProperty(key);
    assertThat(supplier.get(), is(defaultValue));
  }

  private void propertyWithValueReturnsExpected(Supplier<Boolean> supplier, String key, String value, boolean expectedValue) {
    setProperty(key, value);
    assertThat(supplier.get(), is(expectedValue));
  }

}
