/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api.config;

import static com.mulesoft.mule.runtime.gw.api.time.period.Period.millis;
import static com.mulesoft.mule.runtime.gw.api.time.period.Period.seconds;
import static com.mulesoft.mule.runtime.gw.internal.encryption.ConfigurationPropertyEncryptionFactory.createConfigurationPropertyEncrypter;
import static java.lang.Boolean.valueOf;
import static java.lang.System.clearProperty;
import static java.lang.System.getProperty;
import static java.util.Arrays.asList;
import static java.util.Arrays.stream;
import static java.util.Optional.ofNullable;
import static java.util.stream.Collectors.toList;
import static org.apache.commons.lang3.StringUtils.trimToNull;

import org.mule.runtime.api.util.LazyValue;

import com.mulesoft.mule.runtime.gw.api.exception.GatewayConfigurationException;
import com.mulesoft.mule.runtime.gw.api.time.period.Period;
import com.mulesoft.mule.runtime.gw.internal.encryption.ConfigurationPropertyEncrypter;
import com.mulesoft.mule.runtime.gw.internal.encryption.GatewayEncryptionException;

import java.util.List;
import java.util.Optional;

import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class PlatformClientConfiguration extends RuntimeConfiguration {

  private static final Logger LOGGER = LoggerFactory.getLogger(PlatformClientConfiguration.class);

  /**
   * HTTP client connection timeout in milliseconds
   */
  private static final String CONNECT_TIMEOUT = "anypoint.platform.connect_timeout";
  public static final int CONNECT_TIMEOUT_DEFAULT = 10000;

  /**
   * HTTP client read timeout in milliseconds
   */
  private static final String READ_TIMEOUT = "anypoint.platform.read_timeout";
  public static final int READ_TIMEOUT_DEFAULT = 10000;

  /**
   * client secret to be validated against anypoint.platform.coreservice_base_uri
   */
  public static final String CLIENT_SECRET = "anypoint.platform.client_secret";

  /**
   * client id to be validated against anypoint.platform.coreservice_base_uri
   */
  private static final String CLIENT_ID = "anypoint.platform.client_id";

  /**
   * Used to configure if the gateway instance should run in on_prem mode. Setting this flag on turns:
   * anypoint.platform.platform_base_uri and anypoint.platform.coreservice_base_uri to mandatory fields since in "on_prem" mode
   * you should run this solutions inside your organization. Default is false
   */
  private static final String IS_ON_PREM = "anypoint.platform.on_prem";

  /**
   * Amount of time used to determine the interval between each poll for new policies from api platform. Time is expressed in
   * seconds. Default is 60 sec.
   */
  public static final String POLL_APIS_FREQ = "anypoint.platform.poll_policies_freq";
  private static final int POLL_APIS_FREQ_DEFAULT = 60;

  /**
   * Amount of time used to determine the interval between each poll for retrieving contracts to api platform. Time is expressed
   * in seconds. Default is 15 sec.
   */
  public static final String POLL_CLIENTS_FREQ = "anypoint.platform.poll_clients_freq";
  private static final int POLL_CLIENTS_FREQ_DEFAULT = 15;

  /**
   * Amount of time used to determine the interval between each keep alive notification to api platform. Time is expressed in
   * seconds. Default is 60 sec.
   */
  private static final String POLL_KEEP_ALIVE_FREQ = "anypoint.platform.api_keep_alive_freq";
  private static final int POLL_KEEP_ALIVE_FREQ_DEFAULT = 60;

  /**
   * represents the url to which the agent should report its status. E.g.: [https://qa.anypoint.mulesoft.com/apiplatform]
   */
  private static final String PLATFORM_URI = "anypoint.platform.base_uri";
  private static final String PLATFORM_URI_DEFAULT = "https://anypoint.mulesoft.com";

  /**
   * Defines if the client must validate SSL Certificates when communicating with Platform. Default value is true.
   */
  private static final String PLATFORM_ENABLE_SSL_VALIDATION = "anypoint.platform.enable_ssl_validation";
  private static final String PLATFORM_ENABLE_SSL_VALIDATION_DEFAULT = "true";

  /**
   * Amount of time between each connection attempt to the Platform, if the first has failed. Time unit is in seconds. Default
   * value is 15 sec.
   */
  private static final String PLATFORM_INITIALIZATION_RETRY_FREQ = "anypoint.platform.initialization_retry_freq";
  private static final int PLATFORM_INITIALIZATION_RETRY_FREQ_DEFAULT = 15;

  /**
   * represents the url where the gateway should post analytics status and metrics. E.g.:
   * [https://qa.analytics-ingest.anypoint.mulesoft.com]
   */
  private static final String ANALYTICS_URI = "anypoint.platform.analytics_base_uri";

  /**
   * represents the url where the gateway should post usage metrics. E.g.: [https://qa.metrics-ingest.anypoint.mulesoft.com]
   */
  private static final String METRICS_BASE_URI_LEGACY = "anypoint.platform.metrics.base_uri";
  private static final String METRICS_BASE_URI = "anypoint.platform.metrics_base_uri";

  /**
   * The host of your actual proxy. E.g.: [http://myorg.proxy.net]
   */
  private static final String PROXY_HOST = "anypoint.platform.proxy_host";

  /**
   * The port on which you proxy is listening for incoming requests. Default is 80
   */
  private static final String PROXY_PORT = "anypoint.platform.proxy_port";
  private static final int PROXY_PORT_DEFAULT = 80;

  /**
   * The username used to authenticate against your proxy. E.g.: myproxyuser
   */
  private static final String PROXY_USERNAME = "anypoint.platform.proxy_username";

  /**
   * The password used to authenticate against you proxy. E.g.: myproxypass
   */
  private static final String PROXY_PASSWORD = "anypoint.platform.proxy_password";

  /**
   * Whether backoff will be enabled or not, default is "true".
   */
  public static final String BACKOFF = "anypoint.platform.backoff";
  private static final String BACKOFF_DEFAULT = "true";

  /**
   * The status codes considered for disaster recovery in case of an outage.
   */
  public static final String OUTAGE_STATUS_CODES = "anypoint.platform.outage_status_codes";
  public static final List<Integer> DEFAULT_OUTAGE_STATUS_CODES = asList(400, 401, 403, 405, 429, 502, 503);

  /**
   * The query param to send to anypoint platform to inform where the policy should be downloaded from. possible values are
   * 'internal' for 4.3.0 onwards. This is by design no longer configurable.
   */
  private static final String DEFAULT_POLICY_TEMPLATE_SOURCE = "internal"; // CloudFront

  private LazyValue<ConfigurationPropertyEncrypter> encrypter = new LazyValue<>(() -> createConfigurationPropertyEncrypter());

  public Boolean enableSSLValidation() {
    return valueOf(trimToNull(getProperty(PLATFORM_ENABLE_SSL_VALIDATION,
                                          PLATFORM_ENABLE_SSL_VALIDATION_DEFAULT)));
  }

  public Period getConnectTimeout() {
    return millis(parsePositive0IntOrDefault(CONNECT_TIMEOUT, CONNECT_TIMEOUT_DEFAULT));
  }

  public Period getReadTimeout() {
    return millis(parsePositive0IntOrDefault(READ_TIMEOUT, READ_TIMEOUT_DEFAULT));
  }

  public String getProxyHost() {
    return trimToNull(getProperty(PROXY_HOST));
  }

  public int getProxyPort() {
    return parseIntOrDefault(PROXY_PORT, PROXY_PORT_DEFAULT);
  }

  public String getProxyUserName() {
    return trimToNull(getProperty(PROXY_USERNAME));
  }

  public String getProxyPassword() {
    try {
      return encrypter.get().process(trimToNull(getProperty(PROXY_PASSWORD)));
    } catch (GatewayEncryptionException e) {
      throw new GatewayConfigurationException("Proxy Password cannot be decrypted");
    }
  }

  public boolean isOfflineModeEnabled() {
    return trimToNull(getProperty(CLIENT_ID)) == null || trimToNull(getProperty(CLIENT_SECRET)) == null;
  }

  public String getClientId() {
    try {
      return encrypter.get().process(trimToNull(getProperty(CLIENT_ID)));
    } catch (GatewayEncryptionException e) {
      throw new GatewayConfigurationException("Client ID cannot be decrypted");
    }
  }

  public String getClientSecret() {
    try {
      return encrypter.get().process(trimToNull(getProperty(CLIENT_SECRET)));
    } catch (GatewayEncryptionException e) {
      throw new GatewayConfigurationException("Client Secret cannot be decrypted");
    }
  }

  public Period getApisPollFrequency() {
    return seconds(parseIntOrDefault(POLL_APIS_FREQ, POLL_APIS_FREQ_DEFAULT));
  }

  public Period getApiKeepAliveFrequency() {
    return seconds(parseIntOrDefault(POLL_KEEP_ALIVE_FREQ, POLL_KEEP_ALIVE_FREQ_DEFAULT));
  }

  public Period getClientsPollFrequency() {
    return seconds(parseIntOrDefault(POLL_CLIENTS_FREQ, POLL_CLIENTS_FREQ_DEFAULT));
  }

  public Boolean isOnPrem() {
    return valueOf(trimToNull(getProperty(IS_ON_PREM)));
  }

  public String getPlatformUri() {
    return trimToNull(getProperty(PLATFORM_URI, PLATFORM_URI_DEFAULT));
  }

  public String getPolicyTemplateSource() {
    return DEFAULT_POLICY_TEMPLATE_SOURCE;
  }

  public void clearClientSecret() {
    clearProperty(CLIENT_SECRET);
  }

  public int getPlatformInitializationRetryFrequency() {
    return parseIntOrDefault(PLATFORM_INITIALIZATION_RETRY_FREQ, PLATFORM_INITIALIZATION_RETRY_FREQ_DEFAULT);
  }

  public String getAnalyticsUri() {
    return trimToNull(getProperty(ANALYTICS_URI));
  }

  /**
   * @return the base uri for the metrics ingest service
   */
  public String getMetricsBaseUri() {
    return parseProperty(METRICS_BASE_URI_LEGACY)
        .orElseGet(() -> parseProperty(METRICS_BASE_URI).orElseGet(this::getPlatformUri));
  }

  public Boolean backoffEnabled() {
    return valueOf(trimToNull(getProperty(BACKOFF, BACKOFF_DEFAULT)));
  }

  public List<Integer> getOutagesStatusCodes() {
    return parseStatusCodes(trimToNull(getProperty(OUTAGE_STATUS_CODES)));
  }

  private Optional<String> parseProperty(String property) {
    return ofNullable(trimToNull(getProperty(property)));
  }

  private List<Integer> parseStatusCodes(String outageCodes) {
    if (outageCodes == null) {
      return DEFAULT_OUTAGE_STATUS_CODES;
    }

    String[] codesAsStrings = outageCodes.split("\\s*,\\s*");

    try {
      return stream(codesAsStrings).map(Integer::parseInt).collect(toList());
    } catch (NumberFormatException e) {
      LOGGER.error("Could not parse custom outage status codes, value configured: {}", outageCodes);
      return DEFAULT_OUTAGE_STATUS_CODES;
    }
  }
}
