/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.api.config;

import static java.lang.System.getProperty;
import static java.util.Optional.ofNullable;
import static org.apache.commons.lang3.StringUtils.trimToNull;

import java.io.File;
import java.util.Optional;

public class PlatformServicesConfiguration extends RuntimeConfiguration {

  /**
   * Boolean value indicating that the gateway should poll for apis. Default value *true*
   */
  public static final String POLL_APIS_ENABLED = "anypoint.platform.poll_policies_enabled";
  private static final boolean POLL_APIS_ENABLED_DEFAULT = true;


  /**
   * Boolean value indicating that the gateway should poll for contracts. Default value *true*
   */
  public static final String POLL_CLIENTS_ENABLED = "anypoint.platform.poll_clients_enabled";
  private static final boolean POLL_CLIENTS_ENABLED_DEFAULT = true;


  /**
   * Boolean value indicating that the gateway should push keep alive notifications. Default value *true*
   */
  public static final String KEEP_ALIVE_ENABLED = "anypoint.platform.api_keep_alive_enabled";
  private static final boolean KEEP_ALIVE_ENABLED_DEFAULT = true;

  /**
   * When the platform policies poller is disabled the gateway will search the apis information in the folder specified by this
   * system property (in Absolute Path). Api information will be looked under the file name {{apiId}}.json under said folder.
   */
  private static final String APIS_LOCATION = "anypoint.platform.apis_location";

  public boolean getApisPollEnabled() {
    return parseBooleanOrDefault(POLL_APIS_ENABLED, POLL_APIS_ENABLED_DEFAULT);
  }

  public boolean getApiKeepAliveEnabled() {
    return parseBooleanOrDefault(KEEP_ALIVE_ENABLED, KEEP_ALIVE_ENABLED_DEFAULT);
  }

  public boolean getClientsPollEnabled() {
    return parseBooleanOrDefault(POLL_CLIENTS_ENABLED, POLL_CLIENTS_ENABLED_DEFAULT);
  }

  public Optional<File> apisFolder() {
    String apiLocation = getProperty(APIS_LOCATION);
    return ofNullable(trimToNull(apiLocation))
        .map(File::new);
  }

  public ApisInfoSource apisInfoSource() {
    if (getApisPollEnabled()) {
      return ApisInfoSource.POLLING;
    } else if (apisFolder().isPresent()) {
      return ApisInfoSource.FILE_SYSTEM;
    } else {
      return ApisInfoSource.DISABLED;
    }
  }

  public enum ApisInfoSource {
    POLLING, FILE_SYSTEM, DISABLED
  }

}
