/*
 * (c) 2003-2020 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.mule.runtime.gw.encryption;

import static org.hamcrest.CoreMatchers.is;
import static org.hamcrest.CoreMatchers.nullValue;
import static org.hamcrest.MatcherAssert.assertThat;
import static org.junit.rules.ExpectedException.none;
import static org.mockito.Matchers.any;
import static org.mockito.Mockito.when;

import org.mule.encryption.Encrypter;
import org.mule.encryption.exception.MuleEncryptionException;

import com.mulesoft.mule.runtime.gw.internal.encryption.DefaultRuntimeEncrypter;
import com.mulesoft.mule.runtime.gw.internal.encryption.GatewayEncryptionException;
import com.mulesoft.mule.runtime.gw.internal.encryption.RuntimeEncrypter;

import org.junit.Before;
import org.junit.Rule;
import org.junit.Test;
import org.junit.rules.ExpectedException;
import org.junit.runner.RunWith;
import org.mockito.Mock;
import org.mockito.runners.MockitoJUnitRunner;

@RunWith(MockitoJUnitRunner.class)
public class DefaultRuntimeEncrypterTestCase {

  private static final String BASE64_ENCRYPTED = "![ZW5jcnlwdGVkQnl0ZXM=]";
  private static final String ENCRYPTED = "encryptedBytes";
  private static final String ORIGINAL_VALUE = "Geronimooooo";

  @Rule
  public ExpectedException expectedException = none();

  @Mock
  private Encrypter encrypter;

  private RuntimeEncrypter runtimeEncrypter;

  @Before
  public void setUpEncrypter() throws MuleEncryptionException {
    when(encrypter.encrypt(ORIGINAL_VALUE.getBytes())).thenReturn(ENCRYPTED.getBytes());
    when(encrypter.decrypt(ENCRYPTED.getBytes())).thenReturn(ORIGINAL_VALUE.getBytes());

    runtimeEncrypter = new DefaultRuntimeEncrypter(encrypter);
  }

  @Test
  public void successfulSingleValueEncryption() throws MuleEncryptionException {
    String encryptedValue = runtimeEncrypter.encrypt(ORIGINAL_VALUE);

    assertThat(encryptedValue, is(BASE64_ENCRYPTED));
  }

  @Test
  public void successfulSingleValueDecryption() throws MuleEncryptionException {
    String decryptedValue = runtimeEncrypter.decrypt(BASE64_ENCRYPTED);

    assertThat(decryptedValue, is(ORIGINAL_VALUE));
  }

  @Test
  public void alreadyEncryptedValueIsMaintained() throws MuleEncryptionException {
    String encryptedValue = runtimeEncrypter.encrypt(BASE64_ENCRYPTED);

    assertThat(encryptedValue, is(BASE64_ENCRYPTED));
  }

  @Test
  public void decryptingNotEncryptionExpressionDoesNothing() throws MuleEncryptionException {
    String encryptedValue = runtimeEncrypter.decrypt(ORIGINAL_VALUE);

    assertThat(encryptedValue, is(ORIGINAL_VALUE));
  }

  @Test
  public void invalidKeyThrowsExceptionWhenEncrypting() throws MuleEncryptionException {
    when(encrypter.encrypt(any())).thenThrow(new MuleEncryptionException(""));

    expectedException.expect(GatewayEncryptionException.class);
    expectedException.expectMessage("Error trying to encrypt one of the properties");

    runtimeEncrypter.encrypt(ORIGINAL_VALUE);
  }

  @Test
  public void invalidKeyThrowsExceptionWhenDecrypting() throws MuleEncryptionException {
    when(encrypter.decrypt(any())).thenThrow(new MuleEncryptionException(""));

    expectedException.expect(GatewayEncryptionException.class);
    expectedException.expectMessage("Error trying to decrypt " + BASE64_ENCRYPTED);

    runtimeEncrypter.decrypt(BASE64_ENCRYPTED);
  }

  @Test
  public void encryptingNullDoesNotFail() throws MuleEncryptionException {
    String encryptedValue = runtimeEncrypter.encrypt(null);

    assertThat(encryptedValue, nullValue());
  }

  @Test
  public void decryptingNullDoesNotFail() {
    String encryptedValue = runtimeEncrypter.decrypt(null);

    assertThat(encryptedValue, nullValue());
  }
}
