/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.backoff.configuration;


import static java.util.Arrays.asList;

import com.mulesoft.anypoint.backoff.function.BoundedExponentialFunction;
import com.mulesoft.anypoint.backoff.function.Round;
import com.mulesoft.anypoint.backoff.function.dispersion.RangeDispersant;

import java.util.ArrayList;
import java.util.List;
import java.util.function.Function;

/**
 * Exponential Backoff algortithm configuration used to decrease the rate of polling requests to third party services in case of
 * failures.
 * <p>
 * The configuration includes: - a backoff function used to gradually reduce the request rate in case of errors - a backon
 * function used to gradually go back to the normal request rate - the list of status codes that will fire a backoff - for
 * operations that involve a group of requests, the minimum percentage of failed requests to apply backoff
 */
public class BackoffConfiguration {

  public static final double DEFAULT_BACKOFF_BASE = 2.5;
  public static final double DEFAULT_BACKOFF_MULTIPLIER = 14;
  public static final double DEFAULT_BACKOFF_DISPERSION = 0.6;
  public static final int DEFAULT_BACKOFF_STEPS = 5;
  public static final double DEFAULT_BACKON_BASE = 2.5;
  public static final double DEFAULT_BACKON_MULTIPLIER = 14;
  public static final double DEFAULT_BACKON_DISPERSION = 0.3;
  public static final int DEFAULT_BACKON_STEPS = 3;
  public static final double DEFAULT_FAILURE_PERCENTAGE = 100;
  public static final List<Integer> DEFAULT_OUTAGE_STATUS_CODES = asList(400, 401, 403, 405, 429, 502, 503);
  private final int backoffSteps;
  private final int backonSteps;
  private final Function<Integer, Integer> backoffFunction;
  private final Function<Integer, Integer> backonFunction;
  private final List<Integer> statusCodes;
  private final double failurePercentage;
  private final boolean backoffEnabled;
  private boolean fastRecovery;

  protected BackoffConfiguration(Builder builder) {
    backoffSteps = builder.backoffSteps;
    backonSteps = builder.backonSteps;
    backoffFunction = builder.backoffFunction;
    backonFunction = builder.backonFunction;
    statusCodes = builder.statusCodes;
    failurePercentage = builder.failurePercentage;
    fastRecovery = builder.fastRecovery;
    backoffEnabled = builder.backoffEnabled;
  }

  public int backoffSteps() {
    return backoffSteps;
  }

  public int backonSteps() {
    return backonSteps;
  }

  public List<Integer> statusCodes() {
    return backoffEnabled() ? statusCodes : new ArrayList<>();
  }

  public double failurePercentage() {
    return failurePercentage;
  }

  public Function<Integer, Integer> backoff() {
    return backoffFunction;
  }

  public Function<Integer, Integer> backon() {
    return backonFunction;
  }

  public boolean isFastRecovery() {
    return fastRecovery;
  }

  private boolean backoffEnabled() {
    return backoffEnabled;
  }

  public static class Builder {

    private final boolean backoffEnabled;
    private boolean fastRecovery;
    private int backoffSteps;
    private int backonSteps;
    private Function<Integer, Integer> backoffFunction;
    private Function<Integer, Integer> backonFunction;
    private List<Integer> statusCodes;
    private double failurePercentage;

    public Builder(boolean backoffEnabled) {
      this.fastRecovery = false;
      this.statusCodes = DEFAULT_OUTAGE_STATUS_CODES;
      this.backoffSteps = DEFAULT_BACKOFF_STEPS;
      this.backonSteps = DEFAULT_BACKON_STEPS;
      this.failurePercentage = DEFAULT_FAILURE_PERCENTAGE;
      this.backoffEnabled = backoffEnabled;
      backoff(DEFAULT_BACKOFF_BASE, DEFAULT_BACKOFF_MULTIPLIER, DEFAULT_BACKOFF_DISPERSION, DEFAULT_BACKOFF_STEPS);
      backon(DEFAULT_BACKON_BASE, DEFAULT_BACKON_MULTIPLIER, DEFAULT_BACKON_DISPERSION, DEFAULT_BACKON_STEPS);
    }

    public Builder backoff(double base, double multiplier, int steps, Function<Double, Double> dispersant) {
      backoffFunction = new Round(new BoundedExponentialFunction(base, 0, steps, multiplier, dispersant));
      backoffSteps = steps;
      return this;

    }

    public Builder backoff(double base, double multiplier, double dispersion, int steps) {
      return backoff(base, multiplier, steps, new RangeDispersant(dispersion, 1));
    }

    public Builder backon(double base, double multiplier, int steps, Function<Double, Double> dispersant) {
      backonFunction = new Round(new BoundedExponentialFunction(base, 0, steps, multiplier, dispersant));
      backonSteps = steps;
      return this;
    }

    public Builder backon(double base, double multiplier, double dispersion, int steps) {
      return backon(base, multiplier, steps, new RangeDispersant(dispersion, 1));
    }

    public Builder statusCodes(List<Integer> statusCodes) {
      this.statusCodes = statusCodes;
      return this;
    }

    public Builder statusCodes(Integer... statusCodes) {
      return statusCodes(asList(statusCodes));
    }

    public Builder failurePercentage(double failurePercentage) {
      this.failurePercentage = failurePercentage;
      return this;
    }

    public Builder fastRecovery() {
      this.fastRecovery = true;
      return this;
    }

    public BackoffConfiguration build() {
      return new BackoffConfiguration(this);
    }
  }
}
