/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.backoff.function;

import static java.lang.Math.max;
import static java.lang.Math.min;
import static java.util.function.Function.identity;

import java.util.function.Function;

/**
 * Exponential function that has a bounded exponent. The result can be dispersed by a dispersant {@link Function}. If not
 * configured, {@link #identity()} will be used.
 * <p>
 * f(x) = dispersant(m * b ^ x) where b between(minExponent, maxExponent).
 */
public class BoundedExponentialFunction implements Function<Integer, Double> {

  private final double base;
  private final double multiplier;
  private final int minExponent;
  private final int maxExponent;
  private final Function<Double, Double> dispersant;

  public BoundedExponentialFunction(double base,
                                    int minExponent, int maxExponent,
                                    double multiplier) {
    this(base, minExponent, maxExponent, multiplier, identity());
  }


  public BoundedExponentialFunction(double base,
                                    int minExponent, int maxExponent,
                                    double multiplier,
                                    Function<Double, Double> dispersant) {
    this.base = base;
    this.multiplier = multiplier;
    this.minExponent = minExponent;
    this.maxExponent = maxExponent;
    this.dispersant = dispersant;
  }

  @Override
  public Double apply(Integer x) {
    int exponent = max(min(x, maxExponent), minExponent);
    return dispersant.apply(multiplier * Math.pow(base, exponent));
  }

  @Override
  public boolean equals(Object o) {
    if (this == o)
      return true;
    if (o == null || getClass() != o.getClass())
      return false;

    BoundedExponentialFunction that = (BoundedExponentialFunction) o;

    if (Double.compare(that.base, base) != 0)
      return false;
    if (Double.compare(that.multiplier, multiplier) != 0)
      return false;
    if (minExponent != that.minExponent)
      return false;
    if (maxExponent != that.maxExponent)
      return false;
    return dispersant.equals(that.dispersant);
  }

  @Override
  public int hashCode() {
    int result;
    long temp;
    temp = Double.doubleToLongBits(base);
    result = (int) (temp ^ (temp >>> 32));
    temp = Double.doubleToLongBits(multiplier);
    result = 31 * result + (int) (temp ^ (temp >>> 32));
    result = 31 * result + minExponent;
    result = 31 * result + maxExponent;
    result = 31 * result + dispersant.hashCode();
    return result;
  }

  @Override
  public String toString() {
    return "BoundedExponentialFunction{" +
        "base=" + base +
        ", multiplier=" + multiplier +
        ", minExponent=" + minExponent +
        ", maxExponent=" + maxExponent +
        ", dispersant=" + dispersant +
        '}';
  }
}
