/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.backoff.scheduler;

import static java.util.concurrent.TimeUnit.MILLISECONDS;

import org.mule.runtime.api.lifecycle.Disposable;

import com.mulesoft.anypoint.backoff.scheduler.configuration.FastRecoveryConfiguration;
import com.mulesoft.anypoint.backoff.scheduler.configuration.SchedulingConfiguration;
import com.mulesoft.anypoint.backoff.scheduler.observer.FastRecoveryObserver;
import com.mulesoft.anypoint.backoff.scheduler.observer.ReschedulingObserver;
import com.mulesoft.anypoint.backoff.scheduler.runnable.BackoffRunnable;
import com.mulesoft.anypoint.backoff.scheduler.runnable.FastRecovery;
import com.mulesoft.anypoint.backoff.scheduler.runnable.Reschedulable;
import com.mulesoft.anypoint.backoff.state.Stable;
import com.mulesoft.anypoint.backoff.state.Unstable;

import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.concurrent.ScheduledExecutorService;

/**
 * Defines a reduced scheduling interface to run {@link BackoffRunnable}s.
 */
public class BackoffScheduler implements FastRecoveryObserver, ReschedulingObserver, Disposable {

  // TODO: Use SchedulerService (AGW-1285)
  private final ScheduledExecutorService executorService;

  public BackoffScheduler(ScheduledExecutorService executorService) {
    this.executorService = executorService;
  }

  /**
   * Schedules a task in fast recovery mode: {@link BackoffRunnable} will be executed until its first success.
   *
   * @param runnable {@link BackoffRunnable} to be executed.
   * @param configuration configuration parameters.
   * @param recoveryObservers (optional) {@link FastRecoveryObserver}s that will be notified on each execution.
   */
  public void schedule(BackoffRunnable runnable, SchedulingConfiguration configuration,
                       FastRecoveryObserver... recoveryObservers) {
    executorService.schedule(fastRecovery(runnable, configuration, recoveryObservers),
                             configuration.delay().inMillis(),
                             MILLISECONDS);
  }

  /**
   * Schedules a task in rescheduling mode: {@link BackoffRunnable} will be indefinitely scheduled, but with different paces
   * depending on each execution status.
   *
   * @param runnable {@link BackoffRunnable} to be executed.
   * @param configuration configuration parameters.
   */
  public void scheduleWithFixedDelay(BackoffRunnable runnable, SchedulingConfiguration configuration) {
    executorService.schedule(new Reschedulable(runnable, configuration, this),
                             configuration.delay().inMillis(),
                             MILLISECONDS);
  }

  /**
   * {@inheritDoc}
   *
   * {@link BackoffRunnable} will be rescheduled given the new configuration. While this runnable remain {@link Unstable} this
   * process will be repeated.
   */
  @Override
  public FastRecoveryObserver fastRecoveryUnstable(BackoffRunnable runnable, FastRecoveryConfiguration configuration) {
    executorService.schedule(new FastRecovery(runnable, configuration),
                             configuration.delay().inMillis(),
                             MILLISECONDS);
    return this;
  }

  /**
   * {@inheritDoc}
   * 
   * By not doing anything when {@link BackoffRunnable} is {@link Stable}, the {@link BackoffScheduler} will finish the retrying
   * process of a {@link FastRecovery} runnable.
   */
  @Override
  public FastRecoveryObserver fastRecoveryStable(BackoffRunnable runnable, FastRecoveryConfiguration configuration) {
    return this;
  }

  /**
   * {@inheritDoc}
   *
   * By not doing anything when {@link BackoffRunnable} is on {@link Error}, the {@link BackoffScheduler} will finish the retrying
   * process of a {@link FastRecovery} runnable.
   */
  @Override
  public FastRecoveryObserver fastRecoveryAbort(BackoffRunnable runnable) {
    return this;
  }

  /**
   * {@inheritDoc}
   *
   * {@link BackoffRunnable} is rescheduled with new configuration.
   */
  @Override
  public ReschedulingObserver reschedule(BackoffRunnable runnable, SchedulingConfiguration configuration) {
    scheduleWithFixedDelay(runnable, configuration);
    return this;
  }

  @Override
  public void dispose() {
    executorService.shutdownNow();
  }

  private FastRecovery fastRecovery(BackoffRunnable runnable, SchedulingConfiguration schedulingConfiguration,
                                    FastRecoveryObserver... recoveryObservers) {
    return new FastRecovery(runnable, configuration(schedulingConfiguration, recoveryObservers));
  }

  private FastRecoveryConfiguration configuration(SchedulingConfiguration configuration,
                                                  FastRecoveryObserver... recoveryObservers) {
    List<FastRecoveryObserver> allObservers = new ArrayList<>(Arrays.asList(recoveryObservers));
    allObservers.add(this);
    return new FastRecoveryConfiguration(configuration, allObservers);
  }

}
