/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.backoff.scheduler.runnable;

import com.mulesoft.mule.runtime.gw.api.time.period.Period;
import com.mulesoft.anypoint.backoff.configuration.BackoffConfiguration;
import com.mulesoft.anypoint.backoff.session.BackoffBarrier;
import com.mulesoft.anypoint.backoff.session.SessionMetadata;
import com.mulesoft.anypoint.backoff.state.Stable;
import com.mulesoft.anypoint.backoff.state.State;
import com.mulesoft.anypoint.backoff.state.Unstable;
import com.mulesoft.anypoint.backoff.state.Error;

/**
 * Defines a {@link Runnable} that can be delayed depending on the {@link BackoffBarrier} of the execution. It can be configured
 * to execute an outside closure depending on the stability of its internal {@link State}.
 */
public abstract class BackoffRunnable implements Runnable {

  private State currentState;

  public BackoffRunnable(BackoffConfiguration configuration,
                         BackoffBarrier backoffBarrier) {
    this.currentState = new Stable(configuration, backoffBarrier);
  }

  /**
   * Executes this {@link Runnable} once and updates its current {@link State} depending on the {@link SessionMetadata} obtained
   * from the execution.
   */
  @Override
  public void run() {
    SessionMetadata metadata = execute();
    currentState = currentState.next(metadata);
  }

  /**
   * Configures the current {@link State} with an {@link Unstable} closure.
   * 
   * @param closure to execute when {@link Unstable}.
   * @return this.
   */
  public BackoffRunnable ifUnstable(Runnable closure) {
    currentState.ifUnstable(closure);
    return this;
  }

  /**
   * Configures the current {@link State} with an {@link Error} closure.
   *
   * @param closure to execute when {@link Error}.
   * @return this.
   */
  public BackoffRunnable ifError(Runnable closure) {
    currentState.ifError(closure);
    return this;
  }

  /**
   * Configures the current {@link State} with an {@link Stable} closure.
   * 
   * @param closure to execute when {@link Stable}.
   * @return this.
   */
  public BackoffRunnable otherwise(Runnable closure) {
    currentState.otherwise(closure);
    return this;
  }

  /**
   * Delegates its internal {@link State} the execution of the configured closures.
   * 
   * @return this.
   */
  public BackoffRunnable go() {
    currentState.log().go();
    return this;
  }

  /**
   * @return current {@link Period} delay.
   */
  public Period delay() {
    return currentState.delay();
  }

  protected abstract SessionMetadata execute();

  @Override
  public String toString() {
    return "BackoffRunnable{" +
        "hashcode=" + hashCode() +
        ", currentState=" + currentState +
        '}';
  }
}
