/*
 * (c) 2003-2021 MuleSoft, Inc. This software is protected under international copyright law. All use of this software is subject to
 * MuleSoft's Master Subscription Agreement (or other Terms of Service) separately entered into between you and MuleSoft. If such an
 * agreement is not in place, you may not use the software.
 */
package com.mulesoft.anypoint.backoff.scheduler.runnable;

import static com.mulesoft.anypoint.backoff.scheduler.configuration.SchedulingConfiguration.configuration;

import com.mulesoft.anypoint.backoff.scheduler.configuration.SchedulingConfiguration;
import com.mulesoft.anypoint.backoff.scheduler.observer.ReschedulingObserver;

/**
 * Wraps a {@link BackoffRunnable} and notifies {@link ReschedulingObserver} to reschedule with different
 * {@link SchedulingConfiguration} depending on the stability of its wrapped {@link Runnable}.
 */
public class Reschedulable implements Runnable {

  private final BackoffRunnable innerRunnable;
  private final SchedulingConfiguration configuration;
  private final ReschedulingObserver reschedulingObserver;

  public Reschedulable(BackoffRunnable runnable, SchedulingConfiguration configuration,
                       ReschedulingObserver reschedulingObserver) {
    this.innerRunnable = runnable;
    this.configuration = configuration;
    this.reschedulingObserver = reschedulingObserver;
  }

  @Override
  public void run() {
    innerRunnable.run();
    innerRunnable
        .ifUnstable(() -> reschedulingObserver.reschedule(innerRunnable, backoffConfiguration()))
        .ifError(() -> reschedulingObserver.reschedule(innerRunnable, backoffConfiguration()))
        .otherwise(() -> reschedulingObserver.reschedule(innerRunnable, regularConfiguration()))
        .go();
  }

  private SchedulingConfiguration backoffConfiguration() {
    return configuration(innerRunnable.delay(), configuration.frequency());
  }

  private SchedulingConfiguration regularConfiguration() {
    return configuration(configuration.frequency(), configuration.frequency());
  }
}
